/************************************************************
	Modified June 1992 to work with IC Prolog ][
************************************************************/
/* Copyright (C) 1988, Swedish Institute of Computer Science. */

% SICStus Prolog: formatted output

%:- public format/2, format/3.

% format(+Control, +Arguments)
% format(+Stream, +Control, +Arguments)
% Stream Stream
% atom or list of chars Control corresponds roughly to the first argument of
%				the C stdio function printf().
% list or atom Arguments	corresponds to the rest of the printf()
%				arguments
%
% Description: This is intended to be a clone of the Quintus format/[2, 3]
% predicates. See the Quintus manual for more complete documentation.
% The '~N', '~|', '~+' and '~t' control options are not implemented.
% The '~<n>s' control option prints exactly <n> characters.

format(Control, Arguments) :-
	'$ primcatch'('$ format'(Control, Arguments), format(Control, Arguments)).

format(Stream, Control, Arguments) :-
        '$output'(Curr, Stream),
	(
		'$ primcatch'('$ format'(Control, Arguments),
			format(Stream, Control, Arguments))
	->  OK=yes
	;   OK=no
	),
        '$output'(_, Curr), !,
        OK=yes.

'$ format'(Control, Arguments) :-
	(   atom_chars(Control, ControlList)
	;   ControlList=Control
	),
	(   ArgumentList=Arguments
	;   ArgumentList=[Arguments]
	),
	'$output'(Stream, Stream),
	fmt_parse(ArgumentList, SpecList, ControlList, []), !,
	fmt_print(SpecList, 0, 0' , Stream).

fmt_print([], _, _, _).
fmt_print([X|Xs], Tab, Fill, Stream) :- fmt_print(X, Xs, Tab, Fill, Stream).

fmt_print(settab(Arg,Tab0PlusArg,Pos,Tab), Xs, Tab0, Fill, Stream) :- !,
	Tab0PlusArg is Tab0+Arg,
	'$line_position'(Stream, Pos),
	(   Pos>Tab ->
	    nl,
	    putn(Tab, Fill)
	;   Skip is Tab-Pos,
	    putn(Skip, Fill)
	),
	fmt_print(Xs, Tab, Fill, Stream).
fmt_print(fill(Fill), Xs, Tab, _, Stream) :- !,
	fmt_print(Xs, Tab, Fill, Stream).
fmt_print(spec(X,A,N), Xs, Tab, Fill, Stream) :- !,
	'$line_count'(Stream, Lc0),
	fmt_pr(X, A, N),
	'$line_count'(Stream, Lc),
	fmt_print(Lc0, Lc, Xs, Tab, Fill, Stream).
fmt_print(0'\n, Xs, _, _, Stream) :- !,
	nl,
	fmt_print(Xs, 0, 0' , Stream).
fmt_print(C, Xs, Tab, Fill, Stream) :-
	Char is integer(C),
	put(Char),
	fmt_print(Xs, Tab, Fill, Stream).

fmt_print(Lc, Lc, Xs, Tab, Fill, Stream) :- !,
	fmt_print(Xs, Tab, Fill, Stream).
fmt_print(_, _, Xs, _, _, Stream) :- !,
	fmt_print(Xs, 0, 0' , Stream).

fmt_parse([], []) --> [].
fmt_parse(Args, Specs) --> [0'~, C1], !,
	fmt_parse(C1, Args, Specs, 0, D, D).
fmt_parse(Args, Specs) --> [0'\\, 0'c, 0'\n], !,
	fmt_parse(Args, Specs).
fmt_parse(Args, [I|Specs]) --> [I],
	{integer(I)},
	fmt_parse(Args, Specs).

fmt_parse(C, Args, Specs, Sofar, _, D) --> {C>=0'0, C=<0'9}, !,
	{N is 10*Sofar+C-0'0},
	[C1], fmt_parse(C1, Args, Specs, N, N, D).
fmt_parse(0'*, [N|Args], Specs, _, _, D) -->
	{integer(N)},
	[C1], fmt_parse(C1, Args, Specs, 0, N, D).
fmt_parse(0'~, Args, [0'~|Specs], _, 1, 1) -->
	fmt_parse(Args, Specs).
fmt_parse(0'n, Args, [spec(0'c, 0'\n, N)|Specs], _, N, 1) -->
	fmt_parse(Args, Specs).
fmt_parse(0'N, Args, [settab(0,_,_,0)|Specs], _, 1, 1) -->
	fmt_parse(Args, Specs).
fmt_parse(0'|, Args, [Spec|Specs], _, N, current) -->
	(   {current=N} ->
	    {Spec=settab(0,_,Tab,Tab)}
	;   {Spec=settab(N,_,_,N)}
	),
	fmt_parse(Args, Specs).
fmt_parse(0'+, Args, [settab(N,Tab,_,Tab)|Specs], _, N, 8) -->
	fmt_parse(Args, Specs).
fmt_parse(0't, Args, [fill(N)|Specs], _, N, 0' ) --> % faking
	fmt_parse(Args, Specs).
fmt_parse(0'`, Args, [fill(Fill)|Specs], 0, _, _) -->
	[Fill, 0't],
	fmt_parse(Args, Specs).
fmt_parse(0'i, [_|Args], Specs, _, 1, 1) -->
	fmt_parse(Args, Specs).
fmt_parse(0'a, [A|Args], [spec(0'a, A, 1)|Specs], _, 1, 1) -->
	{atom(A)},
	fmt_parse(Args, Specs).
fmt_parse(0'c, [A|Args], [spec(0'c, A, N)|Specs], _, N, 1) -->
	{integer(A)},
	fmt_parse(Args, Specs).
fmt_parse(0'k, [A|Args], [spec(0'k, A, 1)|Specs], _, 1, 1) -->
	fmt_parse(Args, Specs).
fmt_parse(0'p, [A|Args], [spec(0'p, A, 1)|Specs], _, 1, 1) -->
	fmt_parse(Args, Specs).
fmt_parse(0'q, [A|Args], [spec(0'q, A, 1)|Specs], _, 1, 1) -->
	fmt_parse(Args, Specs).
fmt_parse(0'w, [A|Args], [spec(0'w, A, 1)|Specs], _, 1, 1) -->
	fmt_parse(Args, Specs).
fmt_parse(0'e, [A|Args], [spec(0'e, V, N)|Specs], _, N, 6) -->
	{V is float(A)},
	fmt_parse(Args, Specs).
fmt_parse(0'E, [A|Args], [spec(0'E, V, N)|Specs], _, N, 6) -->
	{V is float(A)},
	fmt_parse(Args, Specs).
fmt_parse(0'f, [A|Args], [spec(0'f, V, N)|Specs], _, N, 6) -->
	{V is float(A)},
	fmt_parse(Args, Specs).
fmt_parse(0'g, [A|Args], [spec(0'g, V, N)|Specs], _, N, 6) -->
	{V is float(A)},
	fmt_parse(Args, Specs).
fmt_parse(0'G, [A|Args], [spec(0'G, V, N)|Specs], _, N, 6) -->
	{V is float(A)},
	fmt_parse(Args, Specs).
fmt_parse(0'd, [A|Args], [spec(0'd, V, N)|Specs], _, N, 0) -->
	{V is integer(A)},
	fmt_parse(Args, Specs).
fmt_parse(0'D, [A|Args], [spec(0'D, V, N)|Specs], _, N, 0) -->
	{V is integer(A)},
	fmt_parse(Args, Specs).
fmt_parse(0'r, [A|Args], [spec(0'r, V, N)|Specs], _, N, 8) -->
	{V is integer(A)},
	fmt_parse(Args, Specs).
fmt_parse(0'R, [A|Args], [spec(0'R, V, N)|Specs], _, N, 8) -->
	{V is integer(A)},
	fmt_parse(Args, Specs).
fmt_parse(0's, [A|Args], [spec(0's, A, N)|Specs], _, N, Len) -->
	{is_ascii_list(A, 0, Len)},
	fmt_parse(Args, Specs).

is_ascii_list(X, _, _) :- var(X), !, fail.
is_ascii_list([], N, N).
is_ascii_list([X|Xs], N0, N) :-
	N1 is N0+1,
	integer(X),
	is_ascii_list(Xs, N1, N).

fmt_pr(0'a, Arg, _) :- display(Arg).
fmt_pr(0'k, Arg, _) :- write_canonical(Arg).
fmt_pr(0'p, Arg, _) :- print(Arg).
fmt_pr(0'q, Arg, _) :- writeq(Arg).
fmt_pr(0'w, Arg, _) :- write(Arg).
fmt_pr(0'c, Arg, Number) :-
	putn(Number, Arg).
fmt_pr(0'e, Arg, Number) :-
	'$format_print_float'(0'e, Arg, Number).
fmt_pr(0'E, Arg, Number) :-
	'$format_print_float'(0'E, Arg, Number).
fmt_pr(0'f, Arg, Number) :-
	'$format_print_float'(0'f, Arg, Number).
fmt_pr(0'g, Arg, Number) :-
	'$format_print_float'(0'g, Arg, Number).
fmt_pr(0'G, Arg, Number) :-
	'$format_print_float'(0'G, Arg, Number).
fmt_pr(0'd, Arg, Number) :-
	'$format_print_integer'(0'd, Arg, Number).
fmt_pr(0'D, Arg, Number) :-
	'$format_print_integer'(0'D, Arg, Number).
fmt_pr(0'r, Arg, Number) :-
	'$format_print_integer'(0'r, Arg, Number).
fmt_pr(0'R, Arg, Number) :-
	'$format_print_integer'(0'R, Arg, Number).
fmt_pr(0's, Arg, Number) :-
	putn_list(Number, Arg).

putn(0, _) :- !.
putn(N, C) :-
	N>0, N1 is N-1,
	Char is integer(C),
	put(Char),
	putn(N1, C).

putn_list(0, _) :- !.
putn_list(N, []) :- !,
	N1 is N-1,
	put(0' ),
	putn_list(N1, []).
putn_list(N, [C|Chars]) :-
	N1 is N-1,
	Char is integer(C),
	put(Char),
	putn_list(N1, Chars).

/***************************** ICP utilities **********************************/
'$output'(Curr, Stream) :-
	current_output(Curr),
	set_output(Stream).

'$format_print_float'(A, B, C) :- 'format_print_float%f'(A, B, C).
'$format_print_integer'(A, B, C) :- 'format_print_integer%f'(A, B, C).

'$line_position'(user, Count) :- !, 'line_position%f'(1, Count).
'$line_position'(user_output, Count) :- !, 'line_position%f'(1, Count).
'$line_position'(user_error, Count) :- !, 'line_position%f'(2, Count).
'$line_position'(stream(N), Count) :- !, 'line_position%f'(N, Count).
'$line_position'(_, _) :- 'throw%f'(301).

'$line_count'(user, Count) :- !, 'line_count%f'(1, Count).
'$line_count'(user_output, Count) :- !, 'line_count%f'(1, Count).
'$line_count'(user_error, Count) :- !, 'line_count%f'(2, Count).
'$line_count'(stream(N), Count) :- !, 'line_count%f'(N, Count).
'$line_count'(_, _) :- 'throw%f'(301).

