/* Copyright (C) 1992 Imperial College */
/*  public list :
        [listing/0, listing/1, listfile/1, numbervars/3, portray_clause/1]
*/
/*  The listing/0 and listing/1 commands are based on the Dec-10
    commands, but the format they generate is based on the "pp" command.
    The idea of portray_clause/1 came from PDP-11 Prolog.

    BUG: the arguments of goals are not separated by comma-space but by
    just comma.  This should be fixed, but I haven't the time right not.
    Run the output through COMMA.EM if you really care.

    An irritating fact is that we can't guess reliably which clauses
    were grammar rules, so we can't print them out in grammar rule form.

    We need a proper pretty-printer that takes the line width into
    acount, but it really isn't all that feasible in Dec-10 Prolog.
    Perhaps we could use some ideas from NIL?
*/
listing :-
        '$ pred'(P, A, user, (dynamic)),
        functor(Pred,P,A),
	'$ has_clauses'(Pred),
        nl,
        clause(Pred, Body),
        '$portray_clause$'((Pred:-Body)),
        fail.
listing.


'$ has_clauses'(Pred) :-
	\+ \+ clause(Pred, _), !.


%   listing(PredSpecs)

%   Takes a predicate specifier F/N, a partial specifier F, or a
%   list of such things, and lists each current_predicate Pred
%   matching one of these specifications.

listing(V) :-
        var(V), !.       % ignore variables
listing([]) :- !.
listing([X|Rest]) :- !,
        listing(X),
        listing(Rest).
listing(X) :-
        'functor spec'(X, Name, Low, High),
        '$ pred'(Name, N, user, (dynamic)),
        N >= Low, N =< High,
        functor(Pred, Name, N),
        nl, 
        clause(Pred, Body),
        '$portray_clause$'((Pred:-Body)),
        fail.
listing(_).


'functor spec'(Name/Low-High, Name, Low, High) :- !.
'functor spec'(Name/Arity, Name, Arity, Arity) :- !.
'functor spec'(Name, Name, 0, 255).





listfile(Where) :-
        '$ srcfile'(Where, Name, _),
        open(Name, read, Stream),
        readwrite(Stream),
        close(Stream).

readwrite(Where) :-
        '$ read_stream'(Where,Term,Vars),
        (   Term = (:- Command) ->
            '$ primcatch'(Command, (:- Command))
        ;   Term \== end_of_file
        ),
        test_bind(Vars),
        '$portray_clause$'(Term),
        readwrite(Where).
readwrite(_).

test_bind([]) :- !.
test_bind([X='$VAR'(X)|L]) :-
        test_bind(L).


portray_clause(Call) :-
	copy_term(Call, CopyCall),
	'$portray_clause$'(CopyCall).

'$portray_clause$'(:-(Command)) :-
        (   Command = public(Body), Key = (public)
        ;   Command = mode(Body),   Key = (mode)
        ;   Command = type(Body),   Key = (type)
        ;   Command = pred(Body),   Key = (pred)
        ;   Command = Body,         Key = ''
        ),  !,
        nl,
	numbervars(Body, 0, _),
        'list clauses'(Body, Key, 2, 8).
'$portray_clause$'((Pred:-Body)) :-
	numbervars(Pred+Body, 0, _),
        writeq(Pred),
        'list clauses'(Body, 0, 2, 8), !.
'$portray_clause$'((Pred)) :-
        writeq(Pred),
        'put_atom%f'('.\n').


'list clauses'(V, L, R, D) :-
        var(V), !,
        'list magic'(L, V, D),
        writeq(V),
        'list magic'(R, '.\n').
'list clauses'((A,B), L, R, D) :- !,
        'list clauses'(A, L, 1, D), !,
        'list clauses'(B, 1, R, D).
'list clauses'(true, _, 2, _) :- !,
        'put_atom%f'('.\n').
'list clauses'((A;B), L, R, D) :- !,
        'list magic'(L, fail, D),
        'list magic'((A;B), 0, 2, D),
        'list magic'(R, '.\n').
'list clauses'((A->B), L, R, D) :- !,
        'list clauses'(A, L, 5, D), !,
        'list clauses'(B, 5, R, D).
'list clauses'(Goal, L, R, D) :-
        'list magic'(L, Goal, D),
        writeq(Goal),
        'list magic'(R, '.\n').

'list magic'(0, Goal, D) :- !,
        (   Goal == ! -> 'put_atom%f'(' :- ')
        ;   'put_atom%f'(' :-\n'),
            tab(D)
        ).
'list magic'(1, Goal, D) :- !,
        (   Goal == ! -> 'put_atom%f'(',  ')
        ;   'put_atom%f'(',\n'),
            tab(D)
        ).
'list magic'(3, _, _) :- !,
        'put_atom%f'('(   ').
'list magic'(4, _, _) :- !,
        'put_atom%f'(';   ').
'list magic'(5, _, D) :- !,
        'put_atom%f'(' ->\n'),
        tab(D).
'list magic'(Key, _, D) :-
        atom(Key),
        'put_atom%f'(':- '), write(Key),
        nl, tab(D).


'list magic'(2, C) :- !, 'put_atom%f'(C).
'list magic'(_, _).


'list magic'((A;B), L, R, D) :- !,
        'list magic'(A, L, 1, D), !,
        'list magic'(B, 1, R, D).
'list magic'(Conj,  L, R, D) :-
        'add%f'(D, 8, E),
        'add%f'(L, 3, M),
        'list clauses'(Conj, M, 1, E),
        nl, tab(D),
        'list magic'(R, ')').


/* dac 6/6/90
   made into a primitive for speed
tab(0) :- !.
tab(N) :-
        put(32),
        M is N -- 1,
        tab(M).
*/

numbervars(Term, First, Last) :-
	varsin(Term, List),
	bind_vars(List, First, Last).

bind_vars([], Last, Last).
bind_vars(['$VAR'(First)|T], First, Last) :-
	Next is First + 1,
	bind_vars(T, Next, Last).
