/* Copyright (C) 1992 Imperial College */
/****************************************************************/
/*	Prolog Primitives to support TCP comunications		*/
/****************************************************************/

'$tcp_preds$'([
	'$tcp_init$'/0 = pr_tcp_init,
	'tcp_socket%f'/3 = pr_socket,
	'tcp_setsockopt%f'/3 = pr_setsockopt,
	'tcp_getsockopt%f'/3 = pr_getsockopt,
	'tcp_bind%f'/3 = pr_bind,
	'tcp_listen%f'/1 = pr_listen,
	'tcp_accept%f'/5 = pr_accept,
	'tcp_connect1%f'/3 = pr_connect1,
	'tcp_connect2%f'/2 = pr_connect2,
	'tcp_send%f'/3 = pr_send,
	'tcp_recv%f'/4 = pr_recv,
	'tcp_sendto%f'/5 = pr_sendto,
	'tcp_recvfrom%f'/6 = pr_recvfrom,
	'tcp_sendbr%f'/4 = pr_sendbr,
	'tcp_checkrecv%f'/2 = pr_checkrecv,
	'tcp_checkconn%f'/1 = pr_checkconn,
	'tcp_close%f'/1 = pr_tcp_close,
	'tcp_getsockaddr%f'/3 = pr_getsockaddr,
	'tcp_getpeeraddr%f'/3 = pr_getpeeraddr,
	'tcp_gethost%f'/2 = pr_gethost,
	'tcp_getport%f'/3 = pr_getport,
	'tcp_real_socket%f'/2 = pr_real_socket
]).

/****************************************************************/
/*              High Level Calls                                */
/****************************************************************/

/****************************************************************/
/*	tcp_server(+Port, -Socket)				*/
/*	tcp_server(+Port, -Socket, +Address)			*/
/*		Port: Number					*/
/*		Socket: Integer					*/
/*		Address: Number					*/
/****************************************************************/
tcp_server(Port, Socket) :-
	tcp_open(connection, Socket),
	'$tcp_open$'((tcp_bind(Socket, Port), tcp_listen(Socket)), Socket).

tcp_server(Port, Socket, Address) :-
	tcp_open(connection, Socket),
	'$tcp_open$'((tcp_bind(Socket, Port, Address), tcp_listen(Socket)), Socket).

/****************************************************************/
/*	tcp_client(+Port, +Address, -Socket)			*/
/*	tcp_client(+Port, +Address, -Socket, TimeOut)		*/
/*		Port: Number					*/
/*		Address: Number					*/
/*		Socket: Integer					*/
/*		TimeOut: block, poll, Integer			*/
/****************************************************************/
tcp_client(Port, Address, Socket) :-
	tcp_client(Port, Address, Socket, block).

tcp_client(Port, Address, Socket, TimeOut) :-
	tcp_open(connection, Socket),
	'$tcp_open$'(tcp_connect(Socket, Port, Address, TimeOut), Socket).

/****************************************************************/
/*	tcp_connectionless(+Port, -Socket)			*/
/*		Port: Number					*/
/*		Socket: Integer					*/
/****************************************************************/
tcp_connectionless(Port, Socket) :-
	tcp_open(connectionless, Socket),
	'$tcp_open$'(tcp_bind(Socket, Port), Socket).

/****************************************************************/
/*	tcp_broadcast(+Port, -Socket)				*/
/*		Port: Number					*/
/*		Socket: Integer					*/
/****************************************************************/
tcp_broadcast(Port, Socket) :-
	tcp_open(broadcast, Socket),
	'$tcp_open$'(tcp_bind(Socket, Port), Socket).

/****************************************************************/
/*	Prolog Primitives to support TCP comunications		*/
/****************************************************************/

/****************************************************************/
/*	tcp_open(+Type, -Socket)				*/
/*		Type:	connection, connectionless, broadcast	*/
/*		Socket: Integer					*/
/****************************************************************/
tcp_open(connection, Socket) :-
	tcp_socket(sock_stream, ipproto_ip, Socket).
tcp_open(connectionless, Socket) :-
	tcp_socket(sock_dgram, ipproto_ip, Socket).
tcp_open(broadcast, Socket) :-
	tcp_socket(sock_dgram, ipproto_ip, Socket),
	'$tcp_open$'(tcp_setsockopt(Socket, so_broadcast, 1), Socket).

'$tcp_open$'(Call, _) :-
	call(Call), !.
'$tcp_open$'(_, Socket) :-
	tcp_close(Socket), fail.

/****************************************************************/
/* tcp_socket(+Type, +Protocol, -Socket)			*/
/*	Type: sock_stream, sock_dgram, sock_raw			*/
/*	Protocol: ipproto_ip, ipproto_udp, ipproto_tcp		*/
/*		  ipproto_icmp, ipproto_raw			*/
/* We have commited ourselves to AF_INET family which means the	*/
/* type = SOCK_DGRAM | SOCK_STREAM | SOCK_RAW			*/
/* as far as protocols are concerned the following are allowed	*/
/*	any		-> IPPROTO_IP				*/
/*	SOCK_DGRAM	-> IPPROTO_UDP				*/
/*	SOCK_STREAM	-> IPPROTO_TCP				*/
/*	SOCK_RAW	-> IPPROTO_ICMP | IPPROTO_RAW		*/
/*		socket(AF_INET, type, protocol)			*/
/****************************************************************/
tcp_socket(Type, Protocol, Socket) :-
	'$ primcatch'('$ tcp_socket'(Type, Protocol, Socket1),
			tcp_socket(Type, Protocol, Socket)),
	!, Socket=Socket1.

'$ tcp_socket'(Type,Protocol,Socket) :- 'tcp_socket%f'(Type,Protocol,Socket).

/************************************************************************/
/* tcp_setsockopt(+Socket, +Optname, +Value)				*/
/*	Socket: Integer							*/
/*	Optname: one of following options				*/
/*		so_debug	= turn on debugging info recording	*/
/*		so_reuseaddr	= allow local address reuse		*/
/*		so_keepalive	= keep connections alive		*/
/*		so_dontroute	= just use interface addresses		*/
/*		so_broadcast	= permit sending of broadcast msgs	*/
/*		so_oobinline	= leave received OOB data in line	*/
/*		so_sndbuf	= send buffer size			*/
/*		so_rcvbuf	= receive buffer size			*/
/*		so_sndtimeo	= send timeout				*/
/*		so_rcvtimeo	= receive timeout			*/
/*	Value: Integer							*/
/* For now we only support the SOL_SOCKET level				*/
/* tcp_setsockopt(Sock, Opt, Val) maps onto the C calls			*/
/*	int val = Val, opt = Opt, sock = Sock;				*/
/*	setsockopt(sock, SOL_SOCKET, opt, (char *)&val, sizeof(val));	*/
/************************************************************************/
tcp_setsockopt(Socket, Opt, Value) :-
	'$ primcatch'('$ tcp_setsockopt'(Socket, Opt, Value),
			tcp_setsockopt(Socket, Opt, Value)).

'$ tcp_setsockopt'(Socket, Opt, Value) :-
	'tcp_setsockopt%f'(Socket, Opt, Value).

/************************************************************************/
/* tcp_getsockopt(+Socket, +Optname, -Value)				*/
/*	Socket: Integer							*/
/*	Optname: one of following options				*/
/*		so_debug	= turn on debugging info recording	*/
/*		so_reuseaddr	= allow local address reuse		*/
/*		so_keepalive	= keep connections alive		*/
/*		so_dontroute	= just use interface addresses		*/
/*		so_broadcast	= permit sending of broadcast msgs	*/
/*		so_oobinline	= leave received OOB data in line	*/
/*		so_sndbuf	= send buffer size			*/
/*		so_rcvbuf	= receive buffer size			*/
/*		so_sndtimeo	= send timeout				*/
/*		so_rcvtimeo	= receive timeout			*/
/*		so_error	= get error status and clear		*/
/*		so_type		= get socket type			*/
/*	Value: Integer							*/
/* For now we only support the SOL_SOCKET level				*/
/* tcp_getsockopt(Sock, Opt, Val) maps onto the C calls			*/
/*	int val = Val, opt = Opt, sock = Sock, len = sizeof(val);	*/
/*	getsockopt(sock, SOL_SOCKET, opt, (char *)&val, &len);		*/
/************************************************************************/
tcp_getsockopt(Socket, Opt, Value) :-
	'$ primcatch'('$ tcp_getsockopt'(Socket, Opt, Value),
			tcp_getsockopt(Socket, Opt, Value)).

'$ tcp_getsockopt'(Socket, Opt, Value) :-
	'tcp_getsockopt%f'(Socket, Opt, Value).

/****************************************************************/
/*	tcp_bind(+Socket, +Port)				*/
/*	tcp_bind(+Socket, +Port, +Address)			*/
/*		Socket: Integer					*/
/*		Port: Integer					*/
/*		Address: Float or inaddr_any			*/
/****************************************************************/
tcp_bind(Socket, Port) :-
	'$ primcatch'('$ tcp_bind'(Socket, Port, inaddr_any),
			tcp_bind(Socket, Port)).

tcp_bind(Socket, Port, Address) :-
	'$ primcatch'('$ tcp_bind'(Socket, Port, Address),
			tcp_bind(Socket, Port, Address)).

'$ tcp_bind'(Socket, Port, Address) :- 'tcp_bind%f'(Socket, Port, Address).

/****************************************************************/
/*	tcp_listen(+Socket)					*/
/*		Socket: Integer					*/
/****************************************************************/
tcp_listen(Socket) :-
	'$ primcatch'('$ tcp_listen'(Socket), tcp_listen(Socket)).

'$ tcp_listen'(Sock) :- 'tcp_listen%f'(Sock).

/************************************************************************/
/*	tcp_accept(+Socket, -NewSocket)					*/
/*	tcp_accept(+Socket, -NewSocket, -Port, -Address)		*/
/*	tcp_accept(+Socket, -NewSocket, -Port, -Address, +TimeOut)	*/
/*		Socket: Integer						*/
/*		NewSocket: Integer					*/
/*		Port: Number						*/
/*		Address: Number						*/
/*		TimeOut: block, poll, Number				*/
/************************************************************************/
tcp_accept(Socket, NewSocket) :-
	'$ primcatch'('$ tcp_accept'(Socket,NewSocket1,_,_,block),
			tcp_accept(Socket,NewSocket)), !,
	NewSocket = NewSocket1.

tcp_accept(Socket, NewSocket, Port, Address) :-
	'$ primcatch'('$ tcp_accept'(Socket,NewSocket1,Port1,Address1,block),
			tcp_accept(Socket,NewSocket,Port,Address)), !,
	Address = Address1, Port = Port1, NewSocket = NewSocket1.

tcp_accept(Socket, NewSocket, Port, Address, TimeOut) :-
	'$tcp_time$'(TimeOut, RealTime),
	'$ primcatch'('$ tcp_accept'(Socket,NewSocket1,Port1,Address1,RealTime),
			tcp_accept(Socket,NewSocket,Port,Address,TimeOut)), !,
	Address = Address1, Port = Port1, NewSocket = NewSocket1.

'$ tcp_accept'(Socket, NewSocket, Port, Address, RealTime) :-
	'tcp_accept%f'(Socket, NewSocket, Port, Address, RealTime).

/****************************************************************/
/*	tcp_connect(+Socket, +Port, +Address)			*/
/*	tcp_connect(+Socket, +Port, +Address, +Timeout)		*/
/*		Socket: Integer					*/
/*		Port: Number					*/
/*		Address: Number					*/
/*		TimeOut: block, poll, Number			*/
/****************************************************************/
tcp_connect(Socket, Port, Address) :-
	'$ primcatch'('$ tcp_connect'(Socket, Port, Address, block),
			tcp_connect(Socket, Port, Address)).

tcp_connect(Socket, Port, Address, TimeOut) :-
	'$tcp_time$'(TimeOut, RealTime),
	'$ primcatch'('$ tcp_connect'(Socket, Port, Address, RealTime),
			tcp_connect(Socket, Port, Address, TimeOut)).

'$ tcp_connect'(Socket, Port, Address, RealTime) :-
	'tcp_connect1%f'(Socket, Port, Address),
	'tcp_connect2%f'(Socket, RealTime).

/****************************************************************/
/*	tcp_send(+Socket, +Term)				*/
/*	tcp_send(+Socket, +Term, +EFlag)			*/
/*		Socket: Integer					*/
/*		Term: term					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
tcp_send(Socket, Term) :-
	'$ primcatch'('$ tcp_send'(Socket, Term, normal),
			tcp_send(Socket, Term)).

tcp_send(Socket, Term, EFlag) :-
	'$ primcatch'('$ tcp_send'(Socket, Term, EFlag),
			tcp_send(Socket, Term, EFlag)).

'$ tcp_send'(Socket, Term, EFlag) :- 'tcp_send%f'(Socket, Term, EFlag).

/****************************************************************/
/*	tcp_recv(+Socket, -Term)				*/
/*	tcp_recv(+Socket, -Term, +EFlag)			*/
/*	tcp_recv(+Socket, -Term, +EFlag, +TimeOut)		*/
/*		Socket: Integer					*/
/*		Term: term					*/
/*		EFlag: normal, raw				*/
/*		TimeOut: block, poll, Number			*/
/****************************************************************/
tcp_recv(Socket, Term) :-
	'$ primcatch'('$ tcp_recv'(Socket, Term1, normal, block),
			tcp_recv(Socket, Term)), !,
	Term = Term1.

tcp_recv(Socket, Term, EFlag) :-
	'$ primcatch'('$ tcp_recv'(Socket, Term1, EFlag, block),
			tcp_recv(Socket, Term, EFlag)), !,
	Term = Term1.

tcp_recv(Socket, Term, EFlag, TimeOut) :-
	'$tcp_time$'(TimeOut, RealTime),
	'$ primcatch'('$ tcp_recv'(Socket, Term1, EFlag, RealTime),
			tcp_recv(Socket, Term, EFlag, TimeOut)), !,
	Term = Term1.

'$ tcp_recv'(Socket, Term, EFlag, RealTime) :-
	'tcp_recv%f'(Socket, Term, EFlag, RealTime).

/****************************************************************/
/*	tcp_sendto(+Socket, +Term, +Port, +Address)		*/
/*	tcp_sendto(+Socket, +Term, +Port, +Address, +EFlag)	*/
/*		Socket: Integer					*/
/*		Term: term					*/
/*		Port: Number					*/
/*		Address: Number					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
tcp_sendto(Socket, Term, Port, Address) :-
	'$ primcatch'('$ tcp_sendto'(Socket, Term, Port, Address, normal),
			tcp_sendto(Socket, Term, Port, Address)).

tcp_sendto(Socket, Term, Port, Address, EFlag) :-
	'$ primcatch'('$ tcp_sendto'(Socket, Term, Port, Address, EFlag),
			tcp_sendto(Socket, Term, Port, Address, EFlag)).

'$ tcp_sendto'(Socket, Term, Port, Address, EFlag) :-
	'tcp_sendto%f'(Socket, Term, Port, Address, EFlag).

/************************************************************************/
/*	tcp_recvfrom(+Socket, -Term, -Port, -Address)			*/
/*	tcp_recvfrom(+Socket, -Term, -Port, -Address, +EFlag)		*/
/*	tcp_recvfrom(+Socket, -Term, -Port, -Address, +EFlag, +TimeOut)	*/
/*		Socket: Integer						*/
/*		Term: term						*/
/*		Port: Number						*/
/*		Address: Number						*/
/*		EFlag: normal, raw, peek				*/
/*		TimeOut: block, poll, Number				*/
/************************************************************************/
tcp_recvfrom(Socket, Term, Port, Address) :-
	'$ primcatch'('$ tcp_recvfrom'(Socket, Term1, Port1, Address1, normal,
			block),
		tcp_recvfrom(Socket, Term, Port, Address)), !,
	Term = Term1, Address = Address1, Port = Port1.

tcp_recvfrom(Socket, Term, Port, Address, EFlag) :-
	'$ primcatch'('$ tcp_recvfrom'(Socket, Term1, Port1, Address1, EFlag,
			block),
		tcp_recvfrom(Socket, Term, Port, Address, EFlag)), !,
	Term = Term1, Address = Address1, Port = Port1.

tcp_recvfrom(Socket, Term, Port, Address, EFlag, TimeOut) :-
	'$tcp_time$'(TimeOut, RealTime),
	'$ primcatch'('$ tcp_recvfrom'(Socket, Term1, Port1, Address1, EFlag,
			RealTime),
		tcp_recvfrom(Socket, Term, Port, Address, EFlag, TimeOut)), !,
	Term = Term1, Address = Address1, Port = Port1.

'$ tcp_recvfrom'(Socket, Term, Port, Address, EFlag, RealTime) :-
	'tcp_recvfrom%f'(Socket, Term, Port, Address, EFlag, RealTime).

/****************************************************************/
/*	tcp_sendbr(+Socket, +Term, +Port)			*/
/*	tcp_sendbr(+Socket, +Term, +Port, +EFlag)		*/
/*		Socket: Integer					*/
/*		Term: term					*/
/*		Port: Number					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
tcp_sendbr(Socket, Term, Port) :-
	'$ primcatch'('$ tcp_sendbr'(Socket, Term, Port, normal),
		tcp_sendbr(Socket, Term, Port)).

tcp_sendbr(Socket, Term, Port, EFlag) :-
	'$ primcatch'('$ tcp_sendbr'(Socket, Term, Port, EFlag),
		tcp_sendbr(Socket, Term, Port, EFlag)).

'$ tcp_sendbr'(Socket, Term, Port, EFlag) :-
	'tcp_sendbr%f'(Socket, Term, Port, EFlag).


/****************************************************************/
/*	tcp_checkrecv(+Socket)					*/
/*	tcp_checkrecv(+Socket, +EFlag)				*/
/*		Socket: Integer					*/
/*		EFlag: normal, raw				*/
/****************************************************************/
tcp_checkrecv(Socket) :-
	'$ primcatch'('$ tcp_checkrecv'(Socket,normal), tcp_checkrecv(Socket)).

tcp_checkrecv(Socket, EFlag) :-
	'$ primcatch'('$ tcp_checkrecv'(Socket,EFlag),
		tcp_checkrecv(Socket,EFlag)).

'$ tcp_checkrecv'(Socket, EFlag) :- 'tcp_checkrecv%f'(Socket, EFlag).

/****************************************************************/
/*	tcp_checkconn(+Socket)					*/
/*		Socket: Integer					*/
/****************************************************************/
tcp_checkconn(Socket) :-
	'$ primcatch'('$ tcp_checkconn'(Socket), tcp_checkconn(Socket)).

'$ tcp_checkconn'(Socket) :- 'tcp_checkconn%f'(Socket).

/****************************************************************/
/*	tcp_close(+Socket)					*/
/*		Socket: Integer					*/
/****************************************************************/
tcp_close(Socket) :-
	'$ primcatch'('$ tcp_close'(Socket), tcp_close(Socket)).

'$ tcp_close'(Socket) :- 'tcp_close%f'(Socket).

/****************************************************************/
/*	tcp_getsockaddr(+Socket, -Port, -Address)		*/
/*		Socket: Integer					*/
/*		Port: Number					*/
/*		Address: Number					*/
/****************************************************************/
tcp_getsockaddr(Socket, Port, Address) :-
	'$ primcatch'('$ tcp_getsockaddr'(Socket, Port1, Address1),
			tcp_getsockaddr(Socket, Port, Address)), !,
	Address = Address1, Port = Port1.

'$ tcp_getsockaddr'(Socket, Port, Address) :-
	'tcp_getsockaddr%f'(Socket, Port, Address).

/****************************************************************/
/*	tcp_getpeeraddr(+Socket, -Port, -Address)		*/
/*		Socket: Integer					*/
/*		Port: Number					*/
/*		Address: Number					*/
/****************************************************************/
tcp_getpeeraddr(Socket, Port, Address) :-
	'$ primcatch'('$ tcp_getpeeraddr'(Socket, Port1, Address1),
		tcp_getpeeraddr(Socket, Port, Address)), !,
	Address = Address1, Port = Port1.

'$ tcp_getpeeraddr'(Socket, Port, Address) :-
	'tcp_getpeeraddr%f'(Socket, Port, Address).

tcp_currenthost(Name, Address) :- 
	'$ primcatch'('$ tcp_gethost'(Name, Address),
			tcp_currenthost(Name, Address)).

/****************************************************************/
/*	tcp_gethost(?Name, ?Address)				*/
/*		Name: Atom| Var					*/
/*		Address: Float | Var				*/
/*	If Name and Address are both variables then the values	*/
/*	for the host machine are returned			*/
/****************************************************************/
tcp_gethost(Name, Address) :-
	'$ primcatch'('$ tcp_gethost'(Name, Address),
			tcp_gethost(Name, Address)).

'$ tcp_gethost'(Name, Address) :- 'tcp_gethost%f'(Name, Address).

/****************************************************************/
/*	tcp_getport(?Name, ?Proto, ?Port)			*/
/*		Name: Atom					*/
/*		Proto: tcp, udp, default			*/
/*		Port: Number					*/
/****************************************************************/
tcp_getport(Name, Protocol, Port) :-
	'$ primcatch'('$ tcp_getport'(Name, Protocol, Port),
			tcp_getport(Name, Protocol, Port)).

'$ tcp_getport'(Name,Protocol,Port) :- 'tcp_getport%f'(Name,Protocol,Port).

/****************************************************************/
/*	'$tcp_time$'(+TimeOut, -RealTime)			*/
/*		TimeOut: block, poll, Integer			*/
/*		RealTime: block, poll, Number			*/
/****************************************************************/
'$tcp_time$'(block, block) :- !.
'$tcp_time$'(poll, poll) :- !.
'$tcp_time$'(TimeOut, Realtime) :-
	realtime(Now),
	Realtime is TimeOut + Now.

/****************************************************************/
/*	tcp_real_socket(?SocketId, -Socket)			*/
/*		SocketId: Integer				*/
/*		SocketId: Integer				*/
/****************************************************************/
tcp_real_socket(SocketId, Socket) :-
	'$ primcatch'('$ tcp_real_socket'(SocketId, Socket),
			tcp_real_socket(SocketId, Socket)), !.

'$ tcp_real_socket'(SocketId, Socket) :- 'tcp_real_socket%f'(SocketId, Socket).

/*
?- '$tcp_preds$'(L),
   load_foreign('tcp_prolog.o', L),
   '$tcp_init$'.
*/

?- '$tcp_init$'.
