/*  $Id: principal.pl,v 1.1 92/02/25 16:52:08 jan Exp $

    Part of XPCE
    Designed and implemented by Anjo Anjewierden and Jan Wielemaker
    E-mail: jan@swi.psy.uva.nl

    Copyright (C) 1992 University of Amsterdam. All rights reserved.
*/

% External:	chain_list(+@, -)
%		chain_list_object(+@, -)
%		get(+@, +, ...+..., -)
%		get_chain(+@, +, -)
%		get_chain_object(+@, +, -)
%		get_object(+@, +, ...+..., -)
%		new(?@, +)
%		object(+?)
%		object(+?, -)
%		send(+@, +, ...+...)
%		send_list(+@, +)
%		send_list(+@, +, +)

:- op(100, fx, @).
:- op(150, yfx, ?).

%   This file contains the predicates which make PCE-3 available to the
%   Prolog programmer.  The principal predicates define the interface between
%   Prolog and the PCE virtual machine instructions, the remaining predicates
%   are handy short-hands for a variety of common uses.
%
%   The PCE virtual machine contains three instructions: new (to create an
%   object), send (to manipulate an object) and get (to retrieve a value from
%   and object).  These instructions have complementary definitions in the
%   Prolog implementation.  This file assumes the complementary predicates
%   are called:
%
%	'$pce_new'(+@Object, +Description)
%	'$pce_send'(+@Object, +Selector, +Arguments)
%	'$pce_get'(+@Object, +Selector, +Arguments, -Value)
%
%   In addition the following predicates have been defined for convenience:
%
%	'$pce_object'(+@Object)
%	'$pce_object'(+@Object, -Description)
%	'$pce_get_object'(+@Object, +Selector, +Arguments, -Value)
%
%   [Technical note: '$pce_object'/[1, 2] is necessary given the current
%   implementation.  '$pce_get_object' could be defined in terms of '$pce_get'.]
%
%   Predicates which are basically similar except for multiple arguments
%   are hacked around (a little).  These predicates are defined as having at
%   most 10 (ten) arguments, and the arguments are packed in a single term
%   passed to the complementary interface predicates.  For example:
%
%	send(@window, free).
%	send(@view, print, hello).
%	send(@picture, display, circle(50), point(100, 100)).
%
%   become:
%
%	'$pce_send'(@window, free, arguments).
%	'$pce_send'(@view, print, arguments(hello)).
%	'$pce_send'(@picture, display, arguments(circle(50), point(100, 100))).

%   get(+@Object, +Selector, ...+Argument..., -Output)
%
%   Succeeds once if Output is the value returned by invoking get method
%   called Selector on Object.  Returns an object name, except for names, 
%   integers numbers and reals, which are returned as an object description.

get(Obj, Sel, Out) :-
	'$pce_get'(Obj, Sel, arguments, Out).
get(Obj, Sel, A1, Out) :-
	'$pce_get'(Obj, Sel, arguments(A1), Out).
get(Obj, Sel, A1, A2, Out) :-
	'$pce_get'(Obj, Sel, arguments(A1, A2), Out).
get(Obj, Sel, A1, A2, A3, Out) :-
	'$pce_get'(Obj, Sel, arguments(A1, A2, A3), Out).
get(Obj, Sel, A1, A2, A3, A4, Out) :-
	'$pce_get'(Obj, Sel, arguments(A1, A2, A3, A4), Out).
get(Obj, Sel, A1, A2, A3, A4, A5, Out) :-
	'$pce_get'(Obj, Sel, arguments(A1, A2, A3, A4, A5), Out).
get(Obj, Sel, A1, A2, A3, A4, A5, A6, Out) :-
	'$pce_get'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6), Out).
get(Obj, Sel, A1, A2, A3, A4, A5, A6, A7, Out) :-
	'$pce_get'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7), Out).
get(Obj, Sel, A1, A2, A3, A4, A5, A6, A7, A8, Out) :-
	'$pce_get'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7, A8), Out).
get(Obj, Sel, A1, A2, A3, A4, A5, A6, A7, A8, A9, Out) :-
	'$pce_get'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7, A8, A9), Out).
get(Obj, Sel, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, Out) :-
	'$pce_get'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10), Out).


%   get_object(+@Object, +Selector, ...+Argument, ..., -Output)
%
%   Succeeds once if Output is the value returned by invoking get method
%   called Selector on Object.  Output is an object description, except for the
%   special objects @nil, @default, @on and @off all of which are both
%   object descriptions and object names.

get_object(Obj, Sel, Out) :-
	'$pce_get_object'(Obj, Sel, arguments, Out).
get_object(Obj, Sel, A1, Out) :-
	'$pce_get_object'(Obj, Sel, arguments(A1), Out).
get_object(Obj, Sel, A1, A2, Out) :-
	'$pce_get_object'(Obj, Sel, arguments(A1, A2), Out).
get_object(Obj, Sel, A1, A2, A3, Out) :-
	'$pce_get_object'(Obj, Sel, arguments(A1, A2, A3), Out).
get_object(Obj, Sel, A1, A2, A3, A4, Out) :-
	'$pce_get_object'(Obj, Sel, arguments(A1, A2, A3, A4), Out).
get_object(Obj, Sel, A1, A2, A3, A4, A5, Out) :-
	'$pce_get_object'(Obj, Sel, arguments(A1, A2, A3, A4, A5), Out).
get_object(Obj, Sel, A1, A2, A3, A4, A5, A6, Out) :-
	'$pce_get_object'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6), Out).
get_object(Obj, Sel, A1, A2, A3, A4, A5, A6, A7, Out) :-
	'$pce_get_object'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7), Out).
get_object(Obj, Sel, A1, A2, A3, A4, A5, A6, A7, A8, Out) :-
	'$pce_get_object'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7, A8), Out).
get_object(Obj, Sel, A1, A2, A3, A4, A5, A6, A7, A8, A9, Out) :-
	'$pce_get_object'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7, A8, A9), Out).
get_object(Obj, Sel, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, Out) :-
	'$pce_get_object'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10), Out).


%   send(+@Object, +Selector, ...+Arguments...)
%
%   Succeeds if sending a message to Object with Selector and the given
%   Arguments succeeds.

send(Obj, Sel, A1, A2) :-
	'$pce_send'(Obj, Sel, arguments(A1, A2)).
send(Obj, Sel, A1, A2, A3) :-
	'$pce_send'(Obj, Sel, arguments(A1, A2, A3)).
send(Obj, Sel, A1, A2, A3, A4) :-
	'$pce_send'(Obj, Sel, arguments(A1, A2, A3, A4)).
send(Obj, Sel, A1, A2, A3, A4, A5) :-
	'$pce_send'(Obj, Sel, arguments(A1, A2, A3, A4, A5)).
send(Obj, Sel, A1, A2, A3, A4, A5, A6) :-
	'$pce_send'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6)).
send(Obj, Sel, A1, A2, A3, A4, A5, A6, A7) :-
	'$pce_send'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7)).
send(Obj, Sel, A1, A2, A3, A4, A5, A6, A7, A8) :-
	'$pce_send'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7, A8)).
send(Obj, Sel, A1, A2, A3, A4, A5, A6, A7, A8, A9) :-
	'$pce_send'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7, A8, A9)).
send(Obj, Sel, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10) :-
	'$pce_send'(Obj, Sel, arguments(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10)).

send([], _) :-  !.
send(_, []) :- !.
send([Object|Objects], Selectors) :- !, 
	send(Object, Selectors), 
	send(Objects, Selectors).
send(Object, [Selector|Selectors]) :- !, 
	send(Object, Selector), 
	send(Object, Selectors).
send(Object, Selector) :-
	'$pce_send'(Object, Selector, arguments), !.

send([], _,  _) :- !.
send(_, [], _) :- !.
send(_, _, []) :- !.
send([Object|Objects], Selectors, Arguments) :- !, 
	send(Object, Selectors, Arguments), 
	send(Objects, Selectors, Arguments).
send(Objects, [Selector|Selectors], Arguments) :- !, 
	send(Objects, Selector, Arguments), 
	send(Objects, Selectors, Arguments).
send(Object, Selector, [Argument|Arguments]) :- !, 
	send(Object, Selector, Argument), 
	send(Object, Selector, Arguments).
send(Object, Selector, A1) :-
	'$pce_send'(Object, Selector, arguments(A1)), !.

%   get_chain(+@Object, +Selector, -List)
%   get_chain_object(+@Object, +Selector, -List)
%
%   List is a Prolog list constructed from the PCE chain returned by <-Selector
%   on Object.  get_chain/3 returns a list of object names, 
%   get_chain_object/3 returns object descriptions.

get_chain(Object, Selector, List) :-
	get(Object, Selector, Chain), 
	chain_list(Chain, List).

get_chain_object(Object, Selector, List) :-
	get(Object, Selector, Chain), 
	chain_list_object(Chain, List).


%   chain_list(@+Chain, -List)
%   chain_list_object(@+Chain, -List)
%
%   List is a Prolog list of all objects in Chain.  chain_list/2 returns object
%   names, chain_list_object/2 object descriptions.

chain_list(@nil, []) :- !.
chain_list(Chain, List) :-
	get(Chain, class_name, chain), 
	(   send(Chain, current_no, 1)
	->  'chain list 2'(Chain, List)
	;   List = []
	).

'chain list 2'(Chain, [El|Rest]) :-
	get(Chain, next, El), !, 
	'chain list 2'(Chain, Rest).
'chain list 2'(Chain, []) :-
	\+ get(Chain, current, _).

chain_list_object(@nil, []) :- !.
chain_list_object(Chain, List) :-
	get(Chain, class_name, chain), 
	(   send(Chain, current_no, 1)
	->  'chain list object 2'(Chain, List)
	;   List = []
	).

'chain list object 2'(Chain, [El|Rest]) :-
	get_object(Chain, next, El), !, 
	'chain list object 2'(Chain, Rest).
'chain list object 2'(Chain, []) :-
	\+ get(Chain, current, _).


		/********************************
		*             DEFAULTS		*
		********************************/

%	default(+Argument, +Default, -Value)
%	default(+Argument, resource(+Object, +Name), -Value)
%
%	Get the default value for an argument.

default(@default, resource(Obj, Name), Value) :- !, 
	(   get(Obj, resource_value, Name, Value)
	->  true
	;   format(user_error, 
		   'Failed to get resource ~p of ~p~n', [Name, Obj]), 
	    trace, fail
	).
default(@default, Default, Default) :- !.
default(Value,    _Default, Value).
