/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

/*
 *  Word definition and manipulation macros.
 *
 *  A word is a 32 bit integer consisting of a tag field (the top
 *  5 bits), a gc bit and a value field (the bottom 26 bits). 
 *
 *  Various macros are used to define operations on the tags, thus making
 *  it easy to change the format.  ToPtr and ToRef are optimised conversions
 *  for reference types to "real" pointers and vice versa. 
 */
typedef unsigned long	Word;

#define	tag(x)		((Word) (x) & 0xf8000000)
#define	taggc(x)	((Word) (x) & 0xfc000000)
#define	value(x)	((Word) (x) & 0x03ffffff)

/*
 *  tag definitions
 */
#define	REF		0x00000000	/*  reference		*/
#define	UNB		0x80000000	/*  unbound variable	*/
#define STRUCT		0xa0000000	/*  ptr to structure	*/
#define LIST		0xc0000000	/*  ptr to list		*/
#define	ATOM		0xe0000000	/*  atom		*/
#define	FUNCT		0xe8000000	/*  functor		*/
#define	NUMBER		0xf0000000	/*  integer or float	*/
#define	SHORT		0xf8000000	/*  small integer	*/

#define	Zero		0xfa000000	/*  zero as small int   */

/*
 *  word manipulation
 */
#define IsUnb(x)	((long)(x) < (long) STRUCT)
#define	IsRef(x)	((long)(x) > 0)
#define	IsStruct(x)	(tag(x) == STRUCT)
#define	IsList(x)	(tag(x) == LIST)
#define	IsAtom(x)	(tag(x) == ATOM)
#define	IsFunct(x)	(tag(x) == FUNCT)
#define	IsNumber(x)	((Word) (x) >= NUMBER)
#define	IsShort(x)	((Word) (x) >= SHORT)
#define	IsConst(x)	((Word) (x) >= ATOM)

#define AsRef(x)	(Word) (         value(x))
#define AsUnb(x)	(Word) (UNB    | value(x))
#define AsStruct(x)	(Word) (STRUCT | value(x))
#define AsList(x)	(Word) (LIST   | value(x))
#define AsAtom(x)	(Word) (ATOM   | value(x))
#define AsFunct(x)	(Word) (FUNCT  | value(x))
#define AsNumber(x)	(Word) (NUMBER | value(x))
#define AsShort(x)	(Word) (Zero + value(x))

#define PtrVal(x)	((Word *) value(x))
#define UnbVal(x)	((Process *) value(x))
#define StructVal(x)	((Word *) value(x))
#define ListVal(x)	((Word *) value(x))
#define AtomVal(x)	((Atom *) value(x))
#define FunctVal(x)	((Funct *) value(x))
#define ShortVal(x)	((int) ((x) - Zero))

#define	ToPtr(x)	(Word *) (x)
#define	ToRef(x)	(Word) (x)
#define ToShort(x)	(Word) (Zero + (x))

#define	WNULL	(Word *) 0

/*
 *  unification tags, and tag to unify tag conversion
 */
#define U_REF		0		/*  ptr to unbound var	*/
#define U_STRUCT	1		/*  ptr to structure	*/
#define U_LIST		2		/*  ptr to list		*/
#define U_CONST		3		/*  constant types	*/

#define	u_tag(x)	((x >> 29) & 0x3)


/*
 *  "private processor registers"
 */
extern	Word	Reg[];			/*  emulator 'registers'*/
extern	Word	VarTbl[];		/*  suspended var buffer*/
extern	int	Nsuspv;			/*  no. suspended vars	*/

#define	NARGS		64		/*  max no. registers	*/


/*
 *  code, number, atom, functor and procedure definitions
 */
typedef unsigned char	byte;

typedef struct	number	Number;
typedef	struct	atom	Atom;
typedef	struct	funct	Funct;
typedef	struct	proc	Proc;
typedef	struct	op	Op;

#ifdef ALIGN
typedef long	Code;			/*  for word aligned ptrs  */
#else
typedef	short	Code;
#endif

#define WordP(c) 	((Word *)(c))
#define ProcP(c) 	((Proc **)(c))
#define	BuiltinP(c)	((Word (**)())(c))

#define	WordArgSize	(sizeof(Word)/sizeof(Code))

struct	number
{	int	n_type;			/*  int or float	*/
	union int_float
	{	long	N_int;		/*  as long integer	*/
		double	N_float;	/*  as floating pt	*/
	} n_num;
	Number	*n_next;		/*  hash chain		*/
};
#define	n_int	n_num.N_int
#define	n_float	n_num.N_float


struct	atom
{	Atom	*a_next;		/*  hash chain		*/
	Funct	*a_funct;		/*  functor chain 	*/
	Op	*a_ops;			/*  operator info	*/
	struct process	*a_channel;	/*  channel process	*/
	short	a_props;		/*  property flags	*/
	short	a_length;		/*  string length	*/
	char	a_string[1];		/*  start of string	*/
};


struct	funct
{	Atom	*f_name;		/*  functor name	*/
	int	f_arity;		/*  arity		*/
	Funct	*f_next;		/*  functor chain	*/
	Proc	*f_proc;		/*  procedure		*/
};


struct	proc
{	Code	*p_code;		/*  code pointer	*/
	short	p_flags;		/*  procedure flags	*/
	short	p_size;			/*  code size		*/
	Atom	*p_module;		/*  module name		*/
	Proc	*p_next;		/*  procedure chain	*/
};


struct	op
{	Word	op_set;			/*  operator "set"	*/
	short	op_larg[3];		/*  left/self precs	*/
	short	op_rarg[3];		/*  self/right precs	*/
	Op	*op_next;		/*  operator chain	*/
};


#define	N_INT		0		/*  num is an integer	*/
#define	N_FLOAT		1		/*  num is floating pt.	*/

#define IsShortInt(x)	((x) >= -0x2000000 && (x) < 0x2000000)
#define IsShortTag(x)	(taggc(x) == SHORT)

#define IsLong(x)	(tag(x)==NUMBER && ((Number *)value(x))->n_type==N_INT)
#define IsFloat(x)	(tag(x)==NUMBER && ((Number *)value(x))->n_type==N_FLOAT)

#define	LongVal(x)	((Number *) value(x))->n_int
#define	FloatVal(x)	((Number *) value(x))->n_float


#define	H_TBLSIZ	512		/*  hash table size	*/
#define	A_STRSIZ	256		/*  max string length	*/

#define	A_QUOTE		1		/*  atom needs quotes  */
#define	A_ESCAPE	2		/*  atom contains single quote	*/

#define ANULL		(Atom *) 0
#define	string_val(x)	AtomVal(x)->a_string
#define make_atom(x)	AsAtom(findatom(x))


#define	P_PUBLIC	1		/*  procedure is public		*/
#define	P_EXPORT	2		/*  procedure may be imported	*/
#define	P_READONLY	4		/*  procedure is read only	*/

#define TraceOffset	(1 + WordArgSize)

extern	Atom	**a_htable;		/*  atom hash table	*/
extern	Number	**n_htable;		/*  number hash table	*/
extern	Atom	**a_curr_module;	/*  current module name	 */
extern	Code	*c_top, *c_end;		/*  code pointers	*/

extern	Number	*findnumber();
extern	Atom	*findatom();
extern	Funct	*findfunct();
extern	Proc	*findproc();
extern	Code	*alloc_code();
extern	Word	make_int();

extern	char	*sprintf();

/*
 *  the type returned by signal()
 */
#define SIG_T	void
