%** icp_tty
/*******************************************************************************
Copyright (C) 1992, Yannis Cosmadopoulos and Marek Sergot. All rights reserved.
*******************************************************************************/

e(F) :-
	editfile(F),
	set_prop(editor, file, F).

e :-
	get_prop(editor, file, F),
	editfile(F).

menu :-
	user_channels(In, Out),
	make_menu([
		skilaki_consult,
		skilakiQuery,
		forceCompilation,
		changeCompiler,
		clear_told,
		change_undefined,
		abort
	], Menu),
	repeat,
		read_menu(In, Out, Menu, _, Val),
		once(Val),
		fail.

once(Val) :- call(Val), !.

readQuery(Query,VarNames,[IntX,QualX,ProofX,LeashX]) :-
	user_channels(In, Out),
	readQueryMenu([IntX,QualX,ProofX,LeashX]),
	lastQuery(OldQuery,OldVars, _,_,_,_),
	write(Out,'Last query='), writeq(Out,OldQuery),write(Out, '.'),nl(Out),
	repeat,
		write(Out,'Enter Query> '), flush_output(user_output),
		gread(In, Query, VarNames),
	!,
	rememberQuery(Query,VarNames,IntX,QualX,ProofX,LeashX).

readQueryMenu(Options) :-
	user_channels(In, Out),
	lastQuery(_,_, IntX,QualX,ProofX,LeashX),
	CNames=[int,quals,proof,leash],
	getCompilerOptions(CNames,COptions),
	getQueryMenu(CNames, COptions, [IntX,QualX,ProofX,LeashX], Menu, 2),
	read_toggle_menu(In, Out, [m(1,done)|Menu], [_|NewMenu]),
	getQueryMenu(CNames, COptions, Options, NewMenu, 2),
	fixQueryOptions(Options).

getQueryMenu([], [], [], [], _) :- !.
getQueryMenu([_|CNames], [off|COpt], [_|Last], Menu, N) :- !,
	getQueryMenu(CNames, COpt, Last, Menu, N).
getQueryMenu([Name|CNames], [on|COpt], [O|Last], [m(N,Name=O)|Menu],N) :-
	N1 is N + 1, !,
	getQueryMenu(CNames, COpt, Last, Menu, N1).

fixQueryOptions([]).
fixQueryOptions([off|Options]) :- !, fixQueryOptions(Options).
fixQueryOptions([_|Options]) :- fixQueryOptions(Options).

make_menu(A, B) :- make_menu(A, B, 1).

make_menu([], [], _).
make_menu([O|Options], [m(I,O)|Menu], I) :-
	I1 is I + 1,
	make_menu(Options, Menu, I1).

make_toggle_menu(Names, Values, [m(1,done)|Menu]) :-
	make_toggle_menu(Names, Values, Menu, 2).

make_toggle_menu([], [], [], _) :- !.
make_toggle_menu([Name|CNames], [O|Last], [m(N,Name=O)|Menu],N) :-
	N1 is N + 1, !,
	make_toggle_menu(CNames, Last, Menu, N1).

read_toggle_menu(In, Out, Menu, NewMenu) :-
	read_menu(In, Out, Menu, I, Val),
	process_toggle_answer(I, Val, Menu, NewMenu, In, Out).

process_toggle_answer(1, done, Menu, Menu, _, _) :- !.
process_toggle_answer(I, _, Menu, NewMenu, In, Out) :-
	change_toggle_answer(Menu, I, NewMenu1),
	read_toggle_menu(In, Out, NewMenu1, NewMenu).

change_toggle_answer([m(I,Name=Val1)|Menu], I, [m(I,Name=Val2)|Menu]) :- !,
	(Val1=on, Val2=off; Val1=off, Val2=on).
change_toggle_answer([M|Menu], I, [M|NewMenu]) :-
	change_toggle_answer(Menu, I, NewMenu).

read_menu(In, Out, Menu, I, Val) :-
	write_menu(Out, Menu),
	write(Out,'Select one of the above > '), flush_output(user_output),
	read(In, I), nl(Out),
	on(m(I,Val), Menu), !.
read_menu(In, Out, Menu, I, Val) :-
	read_menu(In, Out, Menu, I, Val).

write_menu(_, []).
write_menu(Out, [m(I,N)|Rest]) :-
	tab(Out,3), write(Out,I), tab(Out,2), write(Out,N), nl(Out),
	write_menu(Out, Rest).

skilaki_consult :-
	user_channels(In, Out),
	write(Out,'Please Enter file name :'), flush_output(user_output),
	read(In, File),
	skilaki_consult(File).

skilaki_consult(File1) :-
	user_channels(_, Out),
	fix_file_name(File1, File),
	write(Out,'LOADING '), write(Out,File), nl(Out),
	open(File, read, Fp),
	my_consult(Fp, File),
	close(Fp).
	
fix_file_name(File, File) :-
	concat(_, '.pl', File), !.
fix_file_name(File1, File) :-
	concat(File1, '.pl', File).

reCompile.

anotherSolution(Goal,VarNames,Vars,Quals,Nsol,Proof,_) :-
	user_channels(In, Out),
	lastQuery(Query, QueryVars, _,QualX,ProofX,_),
	write(Out, 'QUERY='), writeq(Out, Query),
	tab(Out, 3), write(Out,'ANSWER number '), write(Out, Nsol), nl(Out),
	write(Out, 'SOLUTION='), writeq(Out, Goal),
	(
		QualX=on, \+ Quals = []
	->
		write(' QUALIFIED'),
		Btn=[qualifications|RB]
	;
		write(' UNQUALIFIED'),
		Btn=RB	
	),
	(ProofX=on -> RB=[proof] ; RB=[]),
	nl(Out),
	write(Out,'------------------------------------------------------------'), nl(Out),
	make_menu([done, 'Another Solution'|Btn], Menu),
	repeat,
		read_menu(In, Out, Menu, _, Choice),
		processAnotherSolutionBtn(Choice, Goal, Vars, VarNames, Proof, Quals, Out),
	!,
	Choice = done.
	
processAnotherSolutionBtn(proof, Goal, Vars, VarNames, Proof, _, Out) :-
	showProof(Out, Goal, Proof, VarNames, Vars), !,
	fail.
processAnotherSolutionBtn(qualifications, Goal, Vars, VarNames, _, Quals, Out) :-
	showQualifications(Out, Goal, Quals, VarNames, Vars), !,
	fail.
processAnotherSolutionBtn(_, _, _, _, _, _, _).

noMoreSolutions(N) :- !,
	lastQuery(_,QueryVars,_,_,_,_),
	noMoreSolutionsText(N, QueryVars, NoWord),
	user_channels(_, Out),
	write(Out,NoWord), nl(Out).

ask_yesno(Template, Answer) :-
	user_channels(In, Out),
	make_menu(['Yes', 'No', 'Don''t know', 'Quit'], Menu),
	write(Out,'Is '), write(Out,Template), write(Out,' true?'), nl(Out),
	read_menu(In, Out, Menu, I, _),
	yesno_button_answer(I, Answer).

yesno_button_answer(1, yes).
yesno_button_answer(2, no).	
yesno_button_answer(3, stop).	
yesno_button_answer(4, _) :- abort.	

read_answers(Goal, Vars, VarNames, Status, Told, Limits, Number, Template, Default, Known) :-
	user_channels(_, Out),
	write(Out, 'Default Answer'=Default), write(Out, '.'), nl(Out),
	set_prop(ask_which, atom_told, []),
	repeat,
		read_answer_button(Goal, VarNames, S1),
		read_more_answers(S1,Goal,Vars,VarNames,Status,Limits,Number,Template, Default, Known),
		\+ Status = read,	
	!,	
	get_prop(ask_which, atom_told, Told).
	
read_more_answers(read,Goal,Vars,_,read,Limits,Number,_, _, _) :-
	user_channels(In, Out),
	write(Out,'Enter Answer> '), flush_output(user_output),
	read(In, Answer),
	check_which_answer(Goal, Answer, Vars, Limits, Number).
read_more_answers(cancel,_,_,_,cancel,_,_,_, _, _) :-
	set_prop(ask_which, atom_told, []), !,
	fail.
read_more_answers(complete,_,_,_,complete,_,_,_, _, _).
read_more_answers(stop,_,_,_,stop,_,_,_, _, _).
read_more_answers(abort,_,_,_,abort,_,_,_, _, _) :-
	abort.
read_more_answers(assumable,_,_,_,assumable,_,_,_, _, _).
read_more_answers(suggest,Goal,Vars,_,suggest,_,_,_, _, _) :-
	user_channels(_, Out),
	get_suggested_answer(Goal, Vars, Suggest),
	get_prop(ask_which, atom_told, Told),
	\+ on(Suggest, Told),
	write(Out,'Suggestion'=Suggest), nl(Out), !,
	fail.
read_more_answers(show,_,_,_,show,_,_,_, _, Known) :-
	user_channels(_, Out),
	get_prop(ask_which, atom_told, Told),
	write(Out,'Told this query:'), nl(Out),
	forall(on(T, Told), (tab(Out,3), write(Out,T), nl(Out))),
	write(Out,'Told previously:'), nl(Out),
	forall(on(T,Known), (tab(Out,3), write(Out,T), nl(Out))),
	write(Out,'----'), nl(Out), !,
	fail.

check_which_answer(Goal, Answer, Vars, Limits, Number) :-
	get_prop(ask_which, atom_told, Told),
	\+ on(Answer, Told),
	check_validity(Answer, Goal, Vars),
	set_prop(ask_which, atom_told, [Answer|Told]),
	length([Answer|Told], Len),
	Answers is Len + Number,
	answered_maximum(Limits, Answers).

read_answer_button(Goal, VarNames, Status) :-
	user_channels(In, Out),
	write(Out,'What '), write(Out,VarNames), write(Out,' make '),
	writeq(Out,Goal), write(Out,' true? (In List form)'),
	nl(Out),
	make_menu(['Add','Cancel','Answers Complete', 'Stop for Now',
		'Assume True', 'Abort','Suggest', 'Show Answers'], Menu),
	read_menu(In, Out, Menu, I, _),
	check_which_button(I, Status).
	
check_which_button(1, read).
check_which_button(2, cancel).
check_which_button(3, complete).
check_which_button(4, stop).
check_which_button(5, assumable).
check_which_button(6, abort).
check_which_button(7, suggest).
check_which_button(8, show).

message(A) :-
	user_channels(_, Out),
	message(A, Out).
	
message([], Out) :-
	nl(Out).
message([H|T], Out) :-
	writeq(Out,H), write(Out,' '),
	message(T, Out).

compilerDialog(Options, NewOptions) :-
	user_channels(In, Out),
	CNames = [error_handler,int,quals,proof,why],
	make_toggle_menu(CNames, Options, Menu),
	read_toggle_menu(In, Out, Menu, NewMenu),
	make_toggle_menu(CNames, NewOptions, NewMenu).

qualifierDialog([XLeash,Xreduce|XRes], [YLeash,Yreduce|YRes]) :-
	user_channels(In, Out),
	leashDialog(XLeash, YLeash, In, Out),
	reverse_choice(Xreduce, NotXreduce),
	ResidueNames = ['none','every resolution','on qualification only',
		'after negation only'],
	current_menu_options(ResidueNames, [NotXreduce|XRes], [ResOn]),
	write(Out,'residue reduction...'), tab(Out,3), write(Out,'(current="'),
	write(Out,ResOn), write(Out,'")'), nl(Out),
	make_menu(ResidueNames, Menu),
	read_menu(In, Out, Menu, _, Val),
	current_menu_options(ResidueNames, [NotYreduce|YRes], [Val]),
	reverse_choice(NotYreduce, Yreduce).

leashDialog(XLeash, YLeash, In, Out) :-
	getCompilerOptions([quals,leash], [on,XLeash]),
	write(Out,'leashed recursion...'), nl(Out),
	make_toggle_menu([leash], [XLeash], Menu),
	read_toggle_menu(In, Out, Menu, NewMenu),
	make_toggle_menu([leash], [YLeash], NewMenu).

stubberDialog(Settings,CurrentStub,Selected) :-
	user_channels(In, Out),
	write(Out,'Options for Undefined'), tab(Out,3), write(Out,'(current="'),
	write(Out,CurrentStub), write(Out,'")'), nl(Out),
	make_menu(Settings, Menu),
	read_menu(In, Out, Menu, _, Selected).

reverse_choice(on,off).
reverse_choice(off,on).

current_menu_options([], [], []).
current_menu_options([N|Names], [on|OnOff], [N|OnNames]) :-
	current_menu_options(Names, OnOff, OnNames).
current_menu_options([_|Names], [off|OnOff], OnNames) :-
	current_menu_options(Names, OnOff, OnNames).

changeUndefinedDialog(Stubs, [Selected]) :-
	user_channels(In, Out),
	write(Out,'Undefined Relations'), nl(Out),
	make_menu(Stubs, Menu),
	read_menu(In, Out, Menu, _, Selected).

displayUnknownInfo(Told, Stop, Pred, Current, StillTold, StillStop, [Selected]) :-
	user_channels(In, Out),
	make_menu([interactive, 'qualifier only', neither], Menu),
	write(Out,'Unknown Pred'=Pred), nl(Out),
	write(Out,'Mode'), tab(Out,3), write(Out,'(current="'),
	write(Out,Current), write(Out,'")'), nl(Out),
	read_menu(In, Out, Menu, _, Selected),
	deleteUnknown(Told, 'Told Values', StillTold),
	deleteUnknown(Stop, 'Stop Conditions', StillStop).

deleteUnknown(List, Text, ListRem) :-
	user_channels(In, Out),
	write(Out,Text), nl(Out),
	make_menu([done|List], Menu),
	read_menu(In, Out, Menu, I, Val),
	\+ I = 1,
	remove(Val, List, List1), !,
	deleteUnknown(List1, Text, ListRem).
deleteUnknown(List, _, List).

forceCompilation :- forceCompile(yes).

forceCompileFiles(yes) :-
	get_cons('DEFINWIN', Files),
	on(File, Files),
	skilaki_consult(File),
	fail.
forceCompileFiles(_).
