%** pi_compiler
/*******************************************************************************
Copyright (C) 1992, Yannis Cosmadopoulos and Marek Sergot. All rights reserved.
*******************************************************************************/

my_consult(Fp, File) :-
	translateFile(Fp, Xrefs, NewClauses, ''),
	housekeep_file(File),
	assertClauses(NewClauses, Preds),
	processUndefined(Xrefs, Stubs), !,
	assertStubs(Stubs),
	set_prop(File, d_allpreds, Preds).
my_consult(_, File) :-
	message(['Failed to compile', File]).

assertClauses([], []) :- !.
assertClauses([pred(Pred/Arity, Clauses)|Rest], [Pred/Arity|Preds]) :-
	skilaki_housekeep(Pred,Arity),
	assertClause(Clauses),
	!,
	assertClauses(Rest, Preds).

assertClause([]) :- !.
assertClause([H|T]) :-
	assert(H), !,
	assertClause(T).

assertStubs([]) :- !.
assertStubs([stub(Pred/Arity, Clauses)|Rest]) :-
	assertClause(Clauses),
	!,
	assertStubs(Rest).

/*
**  translateFile(FpIn, XRefs, NewClauses, LastPred)
**	Performs the file translation.
**	The  argument	XRefs is a tree which holds the predicate
**	names and arities of undefined predicates found in the
**	bodies of clauses in the file.	It has the form
**	tree(Pred/Arity, NewPred/NewArity, L, R)
**	
**	The  argument TClauses is a list of terms of the form
**	pred(Pred/Arity, Clauses)
**	holding the clauses defined in the file.
*/
translateFile(Fp, Xrefs, TClauses, Last) :-
	translateFile1(Fp, Xrefs1, TClauses1, Last),
	flatten(TClauses1, TClauses, []),
	cleanUndefined(Xrefs1, TClauses, Xrefs, []).

/*
**  translateFile1(FpIn, XRefs, NewClauses, LastPred)
**	Performs the file translation.
**	The  argument	XRefs is a tree which holds the predicate
**	names and arities of undefined predicates found in the
**	bodies of clauses in the file.	It has the form
**	tree(Pred/Arity, NewPred/NewArity, L, R)
**	
**	The  argument TClauses is a tree which
**	hold the clauses defined in the file. It has the form
**	tree(Pred/Arity, Clauses, L, R)
*/
translateFile1(Fp, Xrefs, TClauses, Last) :-
	f_vread(Fp, Clause, VN),
	processClause(Clause, VN, Xrefs, TClauses, Last, Next),
	!,
	translateFile1(Fp, Xrefs, TClauses, Next).
translateFile1(_, _, _, _).

/*
**  processClause
**	Determines whether Clause is a directive (not requiring translation).
**	If not, determines the clause index using the predicate list, Preds,
**	and performs the translation.
**	Translated clauses are written out to the output file.
*/
processClause(end_of_file, _, _, _, _, _) :- !,
	fail.
processClause((:- Goal), _, _, _, Current, Current) :- !,
	execute_goal(Goal).
processClause(Clause, VN, Xrefs, TClauses, Last, Current) :-
	translateClause(Clause,Index,TClause,RelArity,Xrefs,VN),
	add_clause(RelArity, TClause, Index, TClauses, Last, Current).
processClause(Clause, _, _, _, Current, Current) :-
	message(['Failed to compile:', Clause]).

add_clause(Key, Clause, Index, _, p(Key, [Clause|Cl1], I), p(Key, Cl2, Index)) :- !,
	Index is I + 1,
	maybeAddChoker(Key, Cl1, Cl2).
add_clause(Key, Clause, Index, _, p(Key, Cl1, I), p(Key, Cl1, Index)) :- !,
	addSame(Cl1, Clause, I, Index, Cl1).
add_clause(Key, Clause, Index, Tree, _, New) :-
	tree_add_clause(Key, Clause, Index, Tree, New).

tree_add_clause(Key, Clause, 1, tree(Key, [Clause|Cl1], _, _), p(Key, Cl2, 1)) :- !,
	maybeAddChoker(Key, Cl1, Cl2).
tree_add_clause(Key, Clause, Index, tree(Key, Cl1, _, _), p(Key, Cl2, Index)) :- !,
	addSame(Cl1, Clause, 1, Index, Cl2).
tree_add_clause(Key, Clause, Index, tree(Key1, _, L, _), New) :-
	Key @< Key1, !,
	tree_add_clause(Key, Clause, Index, L, New).
tree_add_clause(Key, Clause, Index, tree(_, _, _, R), New) :-
	tree_add_clause(Key, Clause, Index, R, New).

addSame([Value|Tail], Value, I, I, Tail) :- !.
addSame([_|T], Value, I, Index, Tail) :-
	I1 is I + 1,
	addSame(T, Value, I1, Index, Tail).

maybeAddChoker(Pred/Arity, [Choker|Cl], Cl) :-
	getCompilerOptions([quals,leash], [on,on]), !,
	chokerClause(Pred, Arity, Choker).
maybeAddChoker(_, Cl, Cl).

resetCompiler :-
	kill(extraArgs),
	argumentForm(Args,Extended,Extras),
	assert((extraArgs(Args,Extended,Extras):-true)), fail.
resetCompiler.

possibleStubberSettings([I,Q,Current], Possible, Use) :-
	possibleStubberSettings1([I,Q],Possible),
	(
		member(Current, Possible)
	->
		Use=Current
	;
		[Use|_]=Possible
	).

possibleStubberSettings1([on,on],[interactive,'qualifier only',neither]).
possibleStubberSettings1([on,off],[interactive,neither]).
possibleStubberSettings1([off,on],['qualifier only',neither]).
possibleStubberSettings1([off,off],[neither]).

compilerDefaults([int=on,quals=on,proof=off,why=off,leash=off,reduce=on,redRes=off,redQual=off,redNeg=on,stubber=interactive,error_handler=on]).

getCompilerOptions(Names, Values) :-
	get_compiler_options(All),
	valuesdict(Names, All, Values).

get_compiler_options(Current) :-
	get_prop(d_compiler,d_options,Setting),
	!,
	Current=Setting.
get_compiler_options(Default) :-
	compilerDefaults(Default),
	set_compiler_options(Default).

setCompilerOptions(Names, Values, Status) :-
	get_compiler_options(Current),
	join_lists(Names, Values, New),
	update_dict(Current, New, Final),
	set_compiler_options(Final),
	compilerChanged(Current, Final, Status).

compilerChanged(Old, New, Status) :- Old=New, !, Status=no.
compilerChanged(_, _, yes).

set_compiler_options(X) :- set_prop(d_compiler,d_options,X).

housekeep_file(File) :-
	get_prop(File, d_allpreds, Preds), !,
	forall(member(Pred/Arity, Preds), skilaki_housekeep(Pred, Arity)).
housekeep_file(_).

skilaki_housekeep(Pred,_) :-
	deleteStubDeclaration(Pred),
	skilakiName(Pred, NewPred),
	kill(NewPred).

execute_goal(Goal) :- call(Goal), !.
execute_goal(Goal) :- message(['Goal failed:', Goal]).

/* change compiler */
changeCompiler :-
	changeCompilerOptions(Change1),
	changeQualifierOptions(Change2),
	changeStubberOptions, !,
	(member(yes,[Change1,Change2])->forceCompile(no);true).

changeCompilerOptions(Change) :-
	CNames = [int,quals,proof,why],
	getCompilerOptions([error_handler|CNames], Options),
	compilerDialog(Options, [ErrY|Options1]), !,
	setCompilerOptions(CNames, Options1, Change),
	setCompilerOptions([error_handler], [ErrY], _),
	maybeKillStubs(ErrY).
changeCompilerOptions(no).

changeQualifierOptions(Change) :-
	CNames = [leash,reduce,redRes,redQual,redNeg],
	getCompilerOptions([quals|CNames], [on|Options]),
	qualifierDialog(Options, Options1), !,
	setCompilerOptions(CNames, Options1, Change).
changeQualifierOptions(no).

changeStubberOptions :-
	getCompilerOptions([int,quals,stubber], Options),
	possibleStubberSettings(Options, Settings,CurrentStub),
	stubberDialog(Settings, CurrentStub, Selected), !,
	setCompilerOptions([stubber], [Selected], _).
changeStubberOptions.

forceCompile(X) :-
	killStubs,
	resetCompiler,
	forceCompileFiles(X).

