/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */
#include "objs.h"
#include "proc.h"
#include "mem.h"
#include "synch.h"

#define codeblksize	(sizeof(m->m_ovflw) / sizeof(Code))
#define clear_segment(m) \
		bzero((char *)m->m_data, sizeof(m->m_data) + sizeof(m->m_ovflw))

extern	int	nsegments;		/*  no. segments to allocate  */

extern	Code	**code_top, **code_end;	/*  code pointers	*/
extern	int	*code_size;		/*  code size		*/

/*
 *  alloc_segment  --  return a new memory segment, either from the local
 *		       free list or else from the global pool.
 */
Mem	*
alloc_segment(type, prev)
int	type;
Mem	*prev;
{
	register Mem	*m;

	if (m_nfree > 0)
	{	m = m_free;
		m_free = m->m_next;
		m_nfree--;
	}
	else
		m = get_new_segments(1);

	m->m_type = type;
	m->m_prev = prev;
	m->m_next = MNULL;
	m->m_top = m->m_data;

	if (prev)
		prev->m_next = m;

	/* argument stack must be initialised */
	if (type & M_STACK)
		clear_segment(m);

	return(m);
}


/*
 *  free_segments  --  return a list of segments to the local free list
 */
free_segments(m)
register Mem	*m;
{
	Mem	*freelist = m_free;

	m_free = m;
	m_nfree++;

	while (m->m_next)
	{	m_nfree++;
		m = m->m_next;
	}

	m->m_next = freelist;
}


/*
 *  get_new_segments  --  get n segments from global pool
 */
Mem	*
get_new_segments(n)
int	n;
{
	register Mem	*m, *mn;
	int	ng, prno, memsize;
	char	*lck;

	ng = n;
	lck = ptrtolck(mem_free);
	lock(lck);
	if (n > *mem_nfree)
		ng = *mem_nfree;
	m = *mem_free;
	*mem_free += ng;
	*mem_nfree -= ng;
	unlock(lck);

	if (ng > 0)
	{	/*  set prno field and chain segments together  */
		prno = PR - Pr0;
		for (mn=m; mn < m+ng-1; mn++)
		{	mn->m_prno = prno;
			mn->m_next = mn+1;
			(mn+1)->m_prev = mn;
		}
		mn->m_prno = prno;
	}

	if (ng < n)
	{	/*
		 *  not enough segments - extend shared memory and try again
		 *  n.b. should use a s/w lock here
		 */
		memsize = nsegments * sizeof(Mem);
		lock(lck);
		if (*mem_nfree == 0)
		{	*mem_free = (Mem *) extend_shmem(memsize, sizeof(Mem));
			*mem_nfree = nsegments;
		}
		unlock(lck);

		if (ng > 0)
		{	/*  got some before - link chains together  */
			mn->m_next = get_new_segments(n - ng);
			mn->m_next->m_prev = mn;
		}
		else
			m = get_new_segments(n - ng);
	}

	return(m);
}


/*
 *  request_segments  --  append (up to) n new memory segments from the local
 *			  free list onto seg "prev", subject to availability.
 */
Mem	*
request_segments(n, type, prev)
int	n, type;
Mem	*prev;
{
	register Mem	*m;
	Mem	*mret;

#ifdef GCTRACE
	printf("%d#request(%d) ", PR-Pr0, n);
#endif

	if (n > m_nfree)
		n = m_nfree;

#ifdef GCTRACE
	printf("=> %d\n", n);
#endif
	if (n <= 0)
		return(MNULL);

	m_nfree -= n;
	mret = m = m_free;
	m->m_prev = prev;
	prev->m_next = m;

	while (--n > 0)
	{	/*  intermediate segments - prev+next fields already okay  */
		m->m_type = type;
		m->m_top = m->m_data;
		m = m->m_next;
		if (type & M_STACK)
			clear_segment(m);
	}

	m->m_type = type;
	m->m_top = m->m_data;
	if (type & M_STACK)
		clear_segment(m);

	m_free = m->m_next;
	m->m_next = MNULL;

	return(mret);
}


/*
 *  get_used_mem  --  from top segment, compute used cells in this data area
 */
get_used_mem(m)
register Mem	*m;
{
	register int u=0;

	while (m)
	{	u += m->m_top - m->m_data;
		m = m->m_prev;
	}

	return(u);
}

/*
 *  get_free_mem  --  from top segment, compute free cells in this data area
 */
get_free_mem(m)
register Mem	*m;
{
	register int f=0;

	while (m)
	{	if (m->m_top < m->m_ovflw)
			f += m->m_ovflw - m->m_top;
		m = m->m_next;
	}

	return(f);
}


/*
 *  alloc_proc  --  get another process area segment
 */
alloc_proc()
{
	register Mem	*m = alloc_proc_segment();

	p_top = (Process *) m->m_data;
	p_end = (Process *) m->m_ovflw + sizeof(m->m_ovflw) / sizeof(Process);
	m->m_top = (Word *) p_end;

	m_pt = m;

	clear_segment(m);
}


/*
 *  alloc_code  --  code area memory allocation
 *		Code space is allocated in blocks of "codesize".
 *		A block is allocated to processor on first demand
 *		or when it has no space left in its current block.
 */
Code	*
alloc_code(n)
int	n;
{
	Code	*c;
	Mem	*m;
	char	*lck;
	int	pgsz, size;

	/*
	 *  If space required is bigger than one segment, we
	 *  need to allocate from system heap.
	 */
	if (n > (sizeof(m->m_data)+sizeof(m->m_ovflw)) / sizeof(Code))
	{
		pgsz = getpagesize();
		size = (n * sizeof(Code) + pgsz-1) & ~ (pgsz-1);
		return((Code *) extend_shmem(size, pgsz));
	}

	while ((c_top + n) >= c_end)
	{	/*
		 *  not enough room in local block, so allocate another
		 *  from global code space.  If new block is not contiguous
		 *  with current one, c_top is reset to start of new block.
		 */
		lck = ptrtolck(code_top);
		lock(lck);
		c = *code_top;
		*code_top += codeblksize;
		*code_size += codeblksize;

		if ((*code_top + codeblksize) >= *code_end)
		{	/* this should be done outside the critical region */
			m = get_new_segments(1);
			*code_top = (Code *) m->m_data;
			*code_end = (Code *) m->m_ovflw;
		}
		unlock(lck);

		if (c != c_end)
			c_top = c;
		c_end = c + codeblksize;
	}

	c = c_top;
	c_top += n;
	return(c);
}



/*
 *  alloc_fgn_code  --  foreign code memory allocation
 *			one segment is reserved for segsz/pgsz blocks
 *			requests for larger blocks served by extend_memory
 */
static	unsigned fgn_free;
static	int	fgn_nfree = 0;

unsigned
alloc_fgn_code(size)
int	size;
{
	int	pgsz = getpagesize();
	unsigned ret;

	size = (size + pgsz-1) & ~ (pgsz-1);

	if (fgn_nfree == 0)
	{	fgn_free = (unsigned) get_new_segments(1);
		fgn_nfree = sizeof(Mem);
	}

	if (size <= fgn_nfree)
	{	ret = fgn_free;
		fgn_free += size;
		fgn_nfree -= size;
	}
	else
		ret = (unsigned) extend_shmem(size, pgsz);

	return(ret);
}

/*
 *  free_fgn_code  --  deallocate block in reserved code segment
 *			must be most recently allocated block.
 */
free_fgn_code(base)
unsigned base;
{
	int	pgsz = getpagesize();

	if ((base + pgsz) == fgn_free)
	{	fgn_free = base;
		fgn_nfree += pgsz;
	}
}
