/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include <stdio.h>
#include <strings.h>
#include <sys/time.h>
#include "objs.h"
#include "mem.h"
#include "macros.h"
#include "ret.h"
#include "stream.h"

int	int_val();
double	float_val();
Word	make_int(), make_float();

extern	Word	atom_lt, atom_eq, atom_gt;
extern	Word	atom_nil;
extern	Word	atom_list, funct_list;
extern	FILE	*user_error;


/*
 *  TEST PREDICATES
 */
bu_var(t1)
register Word	t1;
{
	return(IsRef(t1));
}

bu_nonvar(t1)
register Word	t1;
{
	return(!IsRef(t1));
}

bu_atom(t1)
register Word	t1;
{
	return(IsAtom(t1));
}

bu_integer(t1)
register Word	t1;
{
	if (IsShort(t1))
		return(1);
	if (IsNumber(t1))
		return(IsLong(t1));
	return(0);
}

bu_float(t1)
register Word	t1;
{
	if (IsNumber(t1))
		return(IsFloat(t1));
	return(0);
}

bu_number(t1)
register Word	t1;
{
	return(IsNumber(t1));
}

bu_atomic(t1)
register Word	t1;
{
	return(IsConst(t1));
}


/*
 *  TERM COMPARISON
 */
bu_lex_eq(t1,t2)
register Word	t1,t2;
{
	return( compare(t1, t2) == 0 );
}

bu_lex_ne(t1,t2)
register Word	t1,t2;
{
	return( compare(t1, t2) != 0);
}

bu_lex_lt(t1,t2)
register Word	t1,t2;
{
	return( compare(t1, t2) < 0);
}

bu_lex_le(t1,t2)
register Word	t1,t2;
{
	return( compare(t1, t2) <= 0);
}

bu_lex_gt(t1,t2)
register Word	t1,t2;
{
	return( compare(t1, t2) > 0);
}

bu_lex_ge(t1,t2)
register Word	t1,t2;
{
	return( compare(t1, t2) >= 0);
}

bu_compare(t1,t2,Arg)
register Word	t1,t2,Arg;
{
	int	ret = compare(t1, t2);

	if (ret < 0)
		return( unify(Arg,atom_lt) );
	else if (ret > 0)
		return( unify(Arg,atom_gt) );
	else
		return( unify(Arg,atom_eq) );
}


/*
 *  TERM CONSTRUCTORS and EXTRACTORS
 */
bu_term_to_list(term,list)
Word	term, list;
{
	char	sbuf[A_STRSIZ];
	Word	cdr;
	register char	*s, *s1;
	register Word	*HP;
	double	f;

	/*
	 *  called from name/2, convert integer, float or atom to a list
	 */
	if (IsAtom(term))
		s = string_val(term);
	else if (IsNumber(term))
	{	s = sbuf;

		if (IsShort(term))
			(void) sprintf(s, "%d", ShortVal(term));
		else if (IsLong(term))
			(void) sprintf(s, "%d", LongVal(term));
		else
		{	f = FloatVal(term);
			(void) sprintf(s, "%g", f);
			/*  ensure string contains a decimal point  */
			if (!index(s, '.'))
			{
				if (index(s,'e'))
					(void) sprintf(s, "%.1e", f);
				else	(void) sprintf(s, "%.1f", f);
			}
		}
	}
	else {
		bu_error(term, "name/2: argument should be an atom");
		return(FAIL);
	}

	for (s1 = s; *s1 != '\0'; s1++)
		;

	HP = m_ht->m_top;
	cdr = atom_nil;
	while (s1 > s)
	{	*HP++ = ToShort(*--s1);
		*HP++ = cdr;
		cdr = AsList(HP - 2);
	}
	m_ht->m_top = HP;

	return( unify(cdr, list) );
}

bu_list_to_term(list,term)
Word	list, term;
{
	char	sbuf[A_STRSIZ];
	register Word	car, cdr;
	register char	*s;
	int	atoi();
	double	atof();

	/*
	 *  called from name/2, convert list to an integer, float or atom
	 */
	s = sbuf;
	cdr = list;
	while (cdr != atom_nil)
	{	if ((s - sbuf) >= A_STRSIZ) {
			bu_error(list, "name/2: list exceeds maximum length");
			return(FAIL);
		}
		if (!IsList(cdr))
			return(FAIL);
		car = *ListVal(cdr);
		deref(car);
		if (!IsShort(car))
			return(FAIL);
		*s++ = (char) ShortVal(car);

		cdr = *(ListVal(cdr) + 1);
		deref(cdr);
	}
	*s = '\0';

	/*  determine type of list string  */
	s = sbuf;
	if (*s == '+' || *s == '-')
		s++;
	if (*s >= '0' && *s <= '9')
	{	s++;
		while (*s >= '0' && *s <= '9')
			s++;
		if (*s == '\0')					/*  integer  */
			return( unify(term, make_int(atoi(sbuf))) );

		if (*s == '.')
		{	s++;
			while (*s >= '0' && *s <= '9')
				s++;
		}
		if (*s == 'e' || *s == 'E')
		{	s++;
			if (*s == '+' || *s == '-')
				s++;
			while (*s >= '0' && *s <= '9')
				s++;
		}
		if (*s == '\0')					/*  float  */
			return( unify(term, make_float(atof(sbuf))) );
	}

	return( unify(term, make_atom(sbuf)) );			/*  atom  */
}

bu_list_to_atom(list,term)
Word	list, term;
{
	char	sbuf[A_STRSIZ];
	register Word	car, cdr;
	register char	*s;

	s = sbuf;
	cdr = list;
	while (cdr != atom_nil)
	{	if ((s - sbuf) >= A_STRSIZ) {
			bu_error(list, "atom_chars/2: list exceeds maximum length");
			return(FAIL);
		}
		if (!IsList(cdr))
			return(FAIL);
		car = *ListVal(cdr);
		deref(car);
		if (!IsShort(car))
			return(FAIL);
		*s++ = (char) ShortVal(car);

		cdr = *(ListVal(cdr) + 1);
		deref(cdr);
	}
	*s = '\0';

	return( unify(term, make_atom(sbuf)) );			/*  atom  */
}

bu_st_to_list(term,list)
Word	term, list;
{
	register Word	*p, cdr, *HP;
	int	n;

	/*
	 *  called from =../2, construct list of functor & args from term
	 */
	if (IsStruct(term))
	{	p = StructVal(term);
		n = FunctVal(*p)->f_arity;
		term = AsAtom(FunctVal(*p)->f_name);
		p += n;
	}
	else if (IsList(term))
	{	p = ListVal(term) + 1;
		n = 2;
		term = atom_list;
	}
	else  /*  constant  */
		n = 0;

	HP = m_ht->m_top;
	cdr = atom_nil;
	while (n-- > 0)			/*  functor arguments  */
	{	*HP++ = IsUnb(*p) ? ToRef(p--) : *p--;
		*HP++ = cdr;
		cdr = AsList(HP - 2);
	}
	*HP++ = term;		/*  principal functor / constant  */
	*HP++ = cdr;
	cdr = AsList(HP - 2);
	m_ht->m_top = HP;

	return( unify(cdr, list) );
}

bu_list_to_st(list,term)
Word	list, term;
{
	register Word	*s, *p, name, cdr, *HP;
	int	arity;

	/*
	 *  called from =../2, construct term from list of functor & args
	 */
	if (!IsList(list))
		return(FAIL);
	name = *ListVal(list);
	cdr = *(ListVal(list) + 1);
	deref(name);
	deref(cdr);
	if (!IsConst(name))
		return(FAIL);
	if (cdr == atom_nil)
		return( unify(term, name) );		/*  constant  */

	HP = m_ht->m_top;
	s = HP++;
	arity = 0;
	while (IsList(cdr))
	{	p = ListVal(cdr);
		*HP++ = IsUnb(*p) ? ToRef(p) : *p;
		arity++;

		cdr = *(ListVal(cdr) + 1);
		deref(cdr);
	}
	if (cdr != atom_nil)
		return(FAIL);
	m_ht->m_top = HP;

	*s = AsFunct( findfunct(AtomVal(name), arity) );
	if (*s == funct_list)
		return( unify(term, AsList(s+1)) );	/*  list  */

	return( unify(term, AsStruct(s)) );		/*  structure  */
}

bu_term_to_na(term,name,arity)
Word	term, name, arity;
{
	Word	*p, n, a;

	/*
	 *  called from functor/3, produce functor name & arity from term
	 */
	if (IsStruct(term))
	{	p = StructVal(term);
		n = AsAtom( FunctVal(*p)->f_name );
		a = ToShort( FunctVal(*p)->f_arity );
	}
	else if (IsList(term))
	{	n = atom_list;
		a = ToShort(2);
	}
	else  /*  constant  */
	{	n = term;
		a = ToShort(0);
	}

	return( unify(name, n) && unify(arity, a) );
}

bu_na_to_term(name,arity,term)
Word	name, arity, term;
{
	int	n;
	register Word	*s, *HP;

	/*
	 *  called from functor/3, construct term from functor name & arity
	 */
	if (!IsShort(arity))
		return(FAIL);

	n = ShortVal(arity);
	if (n == 0)
		return( unify(term, name) );		/*  constant  */

	s = HP = m_ht->m_top;
	*HP++ = AsFunct( findfunct(AtomVal(name), n) );

	while (n-- > 0)
		*HP++ = AsUnb(0);
	m_ht->m_top = HP;

	if (*s == funct_list)
		return( unify(term, AsList(s+1)) );	/*  list  */

	return( unify(term, AsStruct(s)) );		/*  structure  */
}

bu_arg(pos,term,arg)
Word	pos, term, arg;
{
	Word	*p;
	int	a, n;

	if (IsStruct(term))
	{	p = StructVal(term);
		a = FunctVal(*p)->f_arity;
		p++;
	}
	else if (IsList(term))
	{	p = ListVal(term);
		a = 2;
	}
	else
		return(FAIL);

	if (!IsShort(pos) || (n = ShortVal(pos)) < 0 || n > a)
		return(FAIL);

	return( unify(arg, ToRef(&p[--n])) );
}

bu_clock(term)
register Word term;
{
	struct timeval tp;
	if (gettimeofday(&tp, NULL) < 0)
		return(FAIL);
	return( unify(term, make_int((int) tp.tv_sec)) );
}


/*
 *  NUMERIC COMPARSIONS
 */
bu_arith_eq(t1,t2)
register Word	t1,t2;
{
	return (t1 == t2);
	/*  comparing int with float?  */
}

bu_arith_ne(t1,t2)
register Word	t1,t2;
{
	return(t1 != t2);
	/*  comparing int with float?  */
}

bu_arith_lt(t1,t2)
register Word	t1,t2;
{
	if (IsShort(t1&t2))
		return (t1 < t2);

	if (IsFloat(t1) || IsFloat(t2))
		return (float_val(t1) < float_val(t2));

	return (int_val(t1) < int_val(t2));
}

bu_arith_le(t1,t2)
register Word	t1,t2;
{
	if (IsShort(t1&t2))
		return (t1 <= t2);

	if (IsFloat(t1) || IsFloat(t2))
		return (float_val(t1) <= float_val(t2));

	return (int_val(t1) <= int_val(t2));
}

bu_arith_gt(t1,t2)
register Word	t1,t2;
{
	if (IsShort(t1&t2))
		return (t1 > t2);

	if (IsFloat(t1) || IsFloat(t2))
		return (float_val(t1) > float_val(t2));

	return (int_val(t1) > int_val(t2));
}

bu_arith_ge(t1,t2)
register Word	t1,t2;
{
	if (IsShort(t1&t2))
		return (t1 >= t2);

	if (IsFloat(t1) || IsFloat(t2))
		return (float_val(t1) >= float_val(t2));

	return (int_val(t1) >= int_val(t2));
}


/*
 *  NUMERIC FUNCTIONS
 */
Word
fu_plus(t1)
register Word	t1;
{
	return(t1);
}

Word
fu_minus(t1)
register Word	t1;
{
	register int    n;

	n = Zero - (t1 - Zero);		/*  try short sub  */
	if (IsShortTag(n))
		return(n);

	if (IsFloat(t1))
		return( make_float(-float_val(t1)) );
	else
		return( make_int(-int_val(t1)) );
}

Word
fu_not(t1)
register Word	t1;
{
	/*  could optimise for short int case here - later!  */

	return( make_int(~int_val(t1)) );
}

Word
fu_integer(t1)
register Word	t1;
{
	if (IsFloat(t1))
		return( make_int(int_val(t1)) );
	return(t1);
}

Word
fu_float(t1)
register Word	t1;
{
	if (IsFloat(t1))
		return(t1);
	return( make_float(float_val(t1)) );
}

Word
fu_add(t1,t2)
register Word   t1,t2;
{
	register int	n, a, b;

	n = t1 + (t2 - Zero);		/*  try short+short add  */
	if (IsShortTag(n))
		return(n);

	if (IsFloat(t1) || IsFloat(t2))
		return( make_float(float_val(t1) + float_val(t2)) );
	else
	{	n = (a=int_val(t1)) + (b=int_val(t2));
		if ((a > 0 && b > 0 && n < 0) ||
		    (a < 0 && b < 0 && n > 0))
			return( make_float(float_val(t1) + float_val(t2)) );
		else
			return( make_int(n) );
	}
}

Word
fu_subtract(t1,t2)
register Word   t1,t2;
{
	register int	n, a, b;

	n = t1 - (t2 - Zero);		/*  try short+short sub  */
	if (IsShortTag(t1&n))
		return(n);

	if (IsFloat(t1) || IsFloat(t2))
		return( make_float(float_val(t1) - float_val(t2)) );
	else
	{	n = (a=int_val(t1)) - (b=int_val(t2));
		if ((a > 0 && b < 0 && n < 0) ||
		    (a < 0 && b > 0 && n > 0))
			return( make_float(float_val(t1) - float_val(t2)) );
		else
			return( make_int(n) );
	}
}

Word
fu_multiply(t1,t2)
register Word   t1,t2;
{
	register int	n, a, b;

	if (IsFloat(t1) || IsFloat(t2))
		return( make_float(float_val(t1) * float_val(t2)) );
	else
	{	n = (a=int_val(t1)) * (b=int_val(t2));
		if (a == n / b)
			return( make_int(n) );
		else
			return( make_float(float_val(t1) * float_val(t2)) );
	}
}

Word
fu_idivide(t1,t2)
register Word   t1,t2;
{
	return( make_int(int_val(t1) / int_val(t2)) );
}

Word
fu_fdivide(t1,t2)
register Word	t1,t2;
{
	return( make_float(float_val(t1) / float_val(t2)) );
}

Word
fu_modulus(t1,t2)
register Word   t1,t2;
{
	return( make_int(int_val(t1) % int_val(t2)) );
}

Word
fu_and(t1,t2)
register Word   t1,t2;
{
	/*  optimise for short case  */
	return( make_int(int_val(t1) & int_val(t2)) );
}

Word
fu_or(t1,t2)
register Word   t1,t2;
{
	/*  optimise for short case  */
	return( make_int(int_val(t1) | int_val(t2)) );
}

Word
fu_lshift(t1,t2)
register Word   t1,t2;
{
	return( make_int(int_val(t1) << int_val(t2)) );
}

Word
fu_rshift(t1,t2)
register Word   t1,t2;
{
	return( make_int(int_val(t1) >> int_val(t2)) );
}


/*
 *  SUPPORT FUNCTIONS
 */

/*
 *  compare two (dereferenced) terms
 *  	returns -ve if t1 @< t2
 *  	returns   0 if t1 == t2
 *  	returns +ve if t1 @> t2
 */
compare(t1,t2)
register Word	t1,t2;
{
	Word	*p1, *p2;
	Funct	*f1, *f2;
	int	cmb_tags, i, arity, ret;

	if (t1 == t2)
		return(0);			/*  equality test  */

#define	CT(t1,t2)	(t1<<2)+t2		/*  combine tags  */

	cmb_tags = CT(u_tag(t1), u_tag(t2));

	switch (cmb_tags)
	{
	case CT(U_REF,U_REF):
		if (t1 < t2)
			return(-1);
		return(1);

	case CT(U_REF,U_STRUCT):
	case CT(U_REF,U_LIST):
	case CT(U_REF,U_CONST):
	case CT(U_CONST,U_STRUCT):
	case CT(U_CONST,U_LIST):
		return(-1);

	case CT(U_STRUCT,U_REF):
	case CT(U_STRUCT,U_CONST):
	case CT(U_LIST,U_CONST):
	case CT(U_LIST,U_REF):
	case CT(U_CONST,U_REF):
		return(1);

	case CT(U_STRUCT,U_STRUCT):
		/*  compare aritys, then functornames, then each arg  */
		p1 = StructVal(t1);
		p2 = StructVal(t2);
		ret = FunctVal(*p2)->f_arity - FunctVal(*p1)->f_arity;
		if (ret != 0)
			return(ret);

		ret = compare(AsAtom(*p1), AsAtom(*p2));
		if (ret != 0)
			return(ret);

		arity = FunctVal(*p1)->f_arity;
		for (i=0; i<arity; i++)
		{	t1 = ToRef(++p1); deref(t1);
			t2 = ToRef(++p2); deref(t2);
			ret = compare(t1,t2);
			if (ret != 0)
				return(ret);
		}
		return(0);

	case CT(U_STRUCT,U_LIST):
		f1 = FunctVal(*StructVal(t1));
		arity = f1->f_arity;
		if (arity < 2)
			return(-1);
		if (arity > 2)
			return(1);
		return( strcmp(f1->f_name->a_string, ".") );

	case CT(U_LIST,U_STRUCT):
		f2 = FunctVal(*StructVal(t2));
		arity = f2->f_arity;
		if (arity < 2)
			return(1);
		if (arity > 2)
			return(-1);
		return( strcmp(".", f2->f_name->a_string) );

	case CT(U_LIST,U_LIST):
		p1 = ListVal(t1);
		p2 = ListVal(t2);
		t1 = ToRef(p1); deref(t1);
		t2 = ToRef(p2); deref(t2);
		ret = compare(t1, t2);
		if (ret != 0)
			return(ret);
		t1 = ToRef(++p1); deref(t1);
		t2 = ToRef(++p2); deref(t2);
		return( compare(t1, t2) );

	case CT(U_CONST,U_CONST):
		if (IsNumber(t1))
		{	if (IsNumber(t2))
			{	if (bu_arith_lt(t1,t2))
					return(-1);
				if (bu_arith_gt(t1,t2))
					return(1);
				return(0);
			}
			return(-1);	/*  t2 is atom  */
		}
		if (IsNumber(t2))
			return(1);	/*  t1 is atom  */

		/*  both atoms  */
		return( strcmp(string_val(t1), string_val(t2)) );
	}

	return(0);	/* keep lint happy ! */
}


int
int_val(n)
Word	n;
{
	if (IsShort(n))
		return(ShortVal(n));
	if (IsLong(n))
		return(LongVal(n));
	return((int) FloatVal(n));
}

double
float_val(n)
Word	n;
{
	if (IsShort(n))
		return((double) ShortVal(n));
	if (IsLong(n))
		return((double) LongVal(n));
	return(FloatVal(n));
}

Word
make_int(i)
int	i;
{
	Number	*n;
	union	int_float num;

	if (IsShortInt(i))
		return(ToShort(i));

	num.N_int = i;
	n = findnumber(N_INT, num);
	return(AsNumber(n));
}


Word
make_float(f)
double	f;
{
	Number	*n;
	union	int_float num;

	num.N_float = f;
	n = findnumber(N_FLOAT, num);
	return(AsNumber(n));
}


bu_arith_error(arg)
Word	arg;
{
	bu_error(arg, "invalid arithmetic expression");
}

bu_error(arg, msg)
Word	arg;
char	*msg;
{
	fprintf(user_error, "[Error: %s: ", msg);
	(void) write_term(st_user_error, arg, 1);
	fprintf(user_error, "]\n");
}
