/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include <stdio.h>
#include "objs.h"
#include "proc.h"
#include "mem.h"
#include "synch.h"
#include "macros.h"
#include "event.h"
#include "instr.h"
#include "ret.h"
#include "trace.h"

Word	Reg[NARGS];			/*  emulator registers	*/
Word	VarTbl[NARGS];			/*  variable table	*/
int	Nsuspv;				/*  no. vars in table	*/


#define	onereg(i)	i = *C++
#define	tworeg(i,j)	i = *C++; j = *C++
#define	oneword(w)	w = *WordP(C), C += WordArgSize

#define READ	0
#define WRITE	1

#undef	return_fail			/*  see assign macro in macros.h  */
#define	return_fail	goto Fail;

#undef	wait_for_argument		/*  replace macro in macros.h  */
#define	wait_for_argument(x)		\
	while (IsRef(x))		\
	{	if (IsUnb(*ToPtr(x)))	\
		{	t1 = x;		\
			goto Suspend;	\
		}			\
		x = (Word) *ToPtr(x);	\
	}


extern	int	Ncalls;
extern	int	Timeslice;
extern	Word	atom_nil;
extern	Word	funct_call1;
extern	Proc	*proc_trace;
extern	Proc	*proc_undefined;
extern	FILE	*user_error;

/*
 *  EXECUTE  --  execute a process
 */
execute()
{
	register Code	*C;		/*  code pointer	*/
	register Word	*A = Reg;	/*  "A" registers	*/
#define	X	A			/*  "X" registers 	*/
	register short	i, j;		/*  register indexes	*/
	register Word	*S;		/*  structure pointer	*/
#define P	((Process *)S)
	register Word	t1,t2;		/*  temp registers	*/
	register Word	*HP,*SP;	/*  heap,stack top ptrs	*/
	Word	*HE,*SE;		/*  heap,stack end ptrs	*/
	Code	*FL;			/*  failure label	*/
	Word	*Y;			/*  environment pointer	*/
	short	refcnt;
	char	*lck;

	VERIFY_ps();
	TRACE_execute();

	PR->pr_state = PR_EXECUTE;

	/*
	 *  initialise registers
	 */
	HP = m_ht->m_top;
	HE = m_ht->m_ovflw;
	SP = m_st->m_top;
	SE = m_st->m_ovflw;

	C = PS->cont;
	FL = PS->nextcl;
	Y = PS->env;

	if (dead(PS->root) || !C)
		goto Fail;

	copyargs(PS->args, Reg, PS->nargs);

	if (Parent_PS != PS->parent)
	{
		Parent_PS = PS->parent;
	}

	/*
	 *  emulate instructions
	 */
	for (;;)
	{
		VERIFY_instr(C,SP,HP);
		TRACE_instr(C);
		TRACE_icount(C);

		switch(*C++)
		{
			/*	HALT!	  */
		case halt:
			*event = E_TERM;
			return;

			/*
			 *	GET INSTRUCTIONS
			 */
		case get_x_variable:
			tworeg(i,j);
			X[i] = A[j];
			break;

		case get_x_value:
			tworeg(i,j);
			if (unify(X[i], A[j]) != SUCCESS)
				goto Fail;
			break;

		case get_y_variable:
			tworeg(i,j);
			Y[i] = A[j];
			break;

		case get_y_value:
			tworeg(i,j);
			if (unify(Y[i], A[j]) != SUCCESS)
				goto Fail;
			break;

		case get_constant:
			onereg(i);
			oneword(t1);

			deref(A[i]);
			if (IsRef(A[i]))
			{	assign(A[i], t1);
			}
			else if (A[i] != t1)
				goto Fail;
			break;

		case get_nil:
			onereg(i);
			deref(A[i]);
			if (IsRef(A[i]))
			{	assign(A[i], atom_nil);
			}
			else if (A[i] != atom_nil)
				goto Fail;
			break;

			/*
			 *	WAIT INSTRUCTIONS
			 */
		case wait_x_value:
			tworeg(i,j);
			if ((i = match(X[i], A[j])) != SUCCESS )
				if (i == SUSPEND)
					goto Suspend2;
				else
					goto Fail;
			break;

		case wait_constant:
			onereg(i);
			oneword(t1);

			wait_for_argument(A[i]);
			if (A[i] != t1)
				goto Fail;
			break;

		case wait_nil:
			onereg(i);
			wait_for_argument(A[i]);
			if (A[i] != atom_nil)
				goto Fail;
			break;

		case wait_structure:
			onereg(i);
			oneword(t1);

			wait_for_argument(A[i]);

			if (IsStruct(A[i]) && *StructVal(A[i]) == t1)
				S = StructVal(A[i]) + 1;
			else
				goto Fail;
			break;

		case wait_list:
			onereg(i);
			wait_for_argument(A[i]);

			if (IsList(A[i])) 
				S = ListVal(A[i]);
			else
				goto Fail;
			break;

		case wait_variable:
			onereg(i);
			wait_for_argument(A[i]);
			break;

		case wait_2_variables:
			tworeg(i,j);
			deref(A[i]);
			deref(A[j]);
			if (IsRef(A[i]) && IsRef(A[j]))
			{	VarTbl[0] = A[i];
				VarTbl[1] = A[j];
				Nsuspv = 2;
				goto Suspend_process;
			}
			break;

			/*
			 *	PUT INSTRUCTIONS
			 */
		case put_x_variable:
			tworeg(i,j);
			A[j] = ToRef(HP);
			X[i] = ToRef(HP);
			*HP++ = AsUnb(0);
			break;

		case put_x_value:
			tworeg(i,j);
			A[j] = X[i];
			break;

		case put_y_variable:
			tworeg(i,j);
			A[j] = ToRef(HP);
			Y[i] = ToRef(HP);
			*HP++ = AsUnb(0);
			break;

		case put_y_value:
			tworeg(i,j);
			A[j] = Y[i];
			break;

		case put_constant:
			onereg(i);
			oneword(A[i]);
			break;

		case put_nil:
			onereg(i);
			A[i] = atom_nil;
			break;

		case put_structure:
			onereg(i);
			A[i] = AsStruct(HP);
			oneword(*HP++);
			break;

		case put_list:
			onereg(i);
			A[i] = AsList(HP);
			break;

			/*
			 *	PUSH INSTRUCTIONS
			 */
		case push_x_variable:
			onereg(i);
			*SP++ = ToRef(HP);
			X[i] = ToRef(HP);
			*HP++ = AsUnb(0);
			break;

		case push_x_value:
			onereg(i);
			*SP++ = X[i];
			break;

		case push_y_variable:
			onereg(i);
			*SP++ = ToRef(HP);
			Y[i] = ToRef(HP);
			*HP++ = AsUnb(0);
			break;

		case push_y_value:
			onereg(i);
			*SP++ = Y[i];
			break;

		case push_constant:
			oneword(*SP++);
			break;

		case push_nil:
			*SP++ = atom_nil;
			break;

		case push_structure:
			*SP++ = AsStruct(HP);
			oneword(*HP++);
			break;

		case push_list:
			*SP++ = AsList(HP);
			break;

			/*
			 *	WRITE INSTRUCTIONS
			 */
		case write_x_variable:
			onereg(i);
			X[i] = ToRef(HP);
			*HP++ = AsUnb(0);
			break;

		case write_x_value:
			onereg(i);
			*HP++ = X[i];
			break;

		case write_constant:
			oneword(t1);
			*HP++ = t1;
			break;

		case write_nil:
			*HP++ = atom_nil;
			break;

		case write_void:
			onereg(i);
			while (i-- > 0)
				*HP++ = AsUnb(0);
			break;

			/*
			 *	READ INSTRUCTIONS
			 */
		case read_x_variable:
			onereg(i);
			X[i] = IsUnb(*S) ? ToRef(S) : *S;
			S++;
			break;

		case read_x_value:
			onereg(i);
			if ((i = match(X[i], ToRef(S))) != SUCCESS )
			{	if (i == SUSPEND)
					goto Suspend2;
				else
					goto Fail;
			}
			S++;
			break;

		case read_constant:
			oneword(t1);

			if ((i = match(t1, ToRef(S))) != SUCCESS )
			{	if (i == SUSPEND)
					goto Suspend2;
				else
					goto Fail;
			}
			S++;
			break;

		case read_nil:
			if ((i = match(ToRef(S), atom_nil)) != SUCCESS )
			{	if (i == SUSPEND)
					goto Suspend2;
				else
					goto Fail;
			}
			S++;
			break;

		case read_void:
			onereg(i);
			S += i;
			break;

			/*
			 *	COMBINATION INSTRUCTIONS
			 *
			 *   NB: in write mode these can be treated as "atomic"
			 *	 and thus safe to use for output unification.
			 */
		case get_list_var_var:
			/*  get_list + unify_x_variable + unify_x_variable  */
			tworeg(i,j);
			t1 = A[i];
			deref(t1);
			if (IsRef(t1))
			{	/*  write mode  */
if (PS->nargs>0) err(0,"get_list_var_var in write mode in input matching!\n");
				S = HP;

				X[j] = ToRef(HP);
				*HP++ = AsUnb(0);

				onereg(j);
				X[j] = ToRef(HP);
				*HP++ = AsUnb(0);

				assign(t1, AsList(S));
			}
			else if (IsList(t1)) 
			{	/*  read mode  */
				S = ListVal(t1);

				X[j] = IsUnb(*S) ? ToRef(S) : *S;
				S++;

				onereg(j);
				X[j] = IsUnb(*S) ? ToRef(S) : *S;
				S++;
			}	
			else
				goto Fail;
			break;

		case get_list_var_val:
			/*  get_list + unify_x_variable + unify_x_value  */
			tworeg(i,j);
			t1 = A[i];
			deref(t1);
			if (IsRef(t1))
			{	/*  write mode  */
				S = HP;

				X[j] = ToRef(HP);
				*HP++ = AsUnb(0);

				onereg(j);
				*HP++ = X[j];

				assign(t1, AsList(S));
			}
			else if (IsList(t1)) 
			{	/*  read mode  */
				S = ListVal(t1);

				X[j] = IsUnb(*S) ? ToRef(S) : *S;
				S++;

				onereg(j);
				if (unify(X[j], ToRef(S)) != SUCCESS)
					goto Fail;
				S++;
			}
			else
				goto Fail;
			break;

		case get_list_val_var:
			/*  get_list + unify_x_value + unify_x_variable  */
			tworeg(i,j);
			t1 = A[i];
			deref(t1);
			if (IsRef(t1))
			{	/*  write mode  */
				S = HP;

				*HP++ = X[j];

				onereg(j);
				X[j] = ToRef(HP);
				*HP++ = AsUnb(0);

				assign(t1, AsList(S));
			}
			else if (IsList(t1)) 
			{	/*  read mode  */
				S = ListVal(t1);

				if (unify(X[j], ToRef(S)) != SUCCESS)
					goto Fail;
				S++;

				onereg(j);
				X[j] = IsUnb(*S) ? ToRef(S) : *S;
				S++;
			}
			else
				goto Fail;
			break;

		case get_list_val_val:
			/*  get_list + unify_x_value + unify_x_value  */
			tworeg(i,j);
			t1 = A[i];
			deref(t1);
			if (IsRef(t1))
			{	/*  write mode  */
				S = HP;

				*HP++ = X[j];

				onereg(j);
				*HP++ = X[j];

				assign(t1, AsList(S));
			}
			else if (IsList(t1)) 
			{	/*  read mode  */
				S = ListVal(t1);

				if (unify(X[j], ToRef(S)) != SUCCESS)
					goto Fail;
				S++;

				onereg(j);
				if (unify(X[j], ToRef(S)) != SUCCESS)
					goto Fail;
				S++;
			}
			else
				goto Fail;
			break;


			/*
			 *	CONTROL INSTRUCTIONS
			 */
		case call:
			onereg(i);
			alloc_ps(P);

			P->parent = PS;
			P->cont = (*ProcP(C))->p_code;
			C += WordArgSize;

			P->args = SP - i;
			P->nargs = i;
			/*  set root if in guard, else inherit it  */
			P->root = (FL) ? PS : PS->root;

			/*  set reference count  */
			if (PS->refcount == 1)
				PS->refcount++;
			else
			{	lck = ptrtolck(PS);
				lock(lck);
				PS->refcount++;
				unlock(lck);
			}
			P->refcount = 1;

			/*  add to runqueue  */
			enqueue_process(P);
			break;

		case call_last:
			onereg(i);

			alloc_ps(P);

			P->parent = PS;

			P->args = WNULL;
			P->nargs = i;
			/*  set root if in parallel guard, else inherit it  */
			P->root = (FL && PS->refcount > 1) ? PS : PS->root;

			/*  set reference count  */
			P->refcount = 1;

			/*  suspend current process, switch to P  */
			Parent_PS = PS;
			PS->cont = (Code *) (ProcP(C) + 1);
			PS = P;
			C = (*ProcP(C))->p_code;
			P->cont = C;
			FL = 0;
			goto Process_switch;

		case call_promoted:
			onereg(i);
			alloc_ps(P);

			P->cont = (*ProcP(C))->p_code;
			C += WordArgSize;

			P->args = SP - i;
			P->nargs = i;
			P->root = PS->root;

			/*  set parent & reference counts  */
			if (Parent_PS)
				Parent_PS->refcount++;
			P->parent = Parent_PS;
			P->refcount = 1;

			/*  add to runqueue  */
			enqueue_process(P);
			break;

		case call_promoted_last:
			onereg(i);

			/*
			 *  current process switches to execute last body goal.
			 *  parent,args,env,refcount and root fields
			 *  of PS should already contain correct values.
			 */
			P = PS;

			P->args = WNULL;
			P->nargs = i;
			C = (*ProcP(C))->p_code;
			P->cont = C;

			/*  fall into... */

		case process_switch:
		Process_switch:
			/*  check state of heap and stack  */
			if (HP > HE)
			{	if (m_ht->m_next)
				{	m_ht->m_top = HP;
					m_ht = m_ht->m_next;
					HP = m_ht->m_data;
					HE = m_ht->m_ovflw;
				}
				else
					*event |= E_GC;
			}
			if (SP > SE)
			{	if (m_st->m_next)
				{	m_st->m_top = SP;
					m_st = m_st->m_next;
					SP = m_st->m_data;
					SE = m_st->m_ovflw;
				}
				else
					*event |= E_SGC;
			}

			if (*event ||
			    (Ncalls > Timeslice && PR->q_front != PR->q_back))
			{	PS->args = SP;
				SP += PS->nargs;
				copyargs(Reg, PS->args, PS->nargs);

				enqueue_process(PS);
                        	m_ht->m_top = HP;
                        	m_st->m_top = SP;
				return;
			}
			break;


		Proceed:
			/*
			 *  (arrive here from enter_c, enter_io_c)
			 *  execute a commit before the proceed instruction
			 */
			if (PS->args != WNULL)
				dealloc_stack(PS->args, PS->nargs);
			Ncalls++;

			/*  fall into....  */

		case proceed:
			/*
			 *  delete current process and
			 *  continue parent if last child
			 */
			P = PS->parent;
			dealloc_ps(PS);

                        m_ht->m_top = HP;
                        m_st->m_top = SP;

			if (!P)
				return;

			/*
			 *  if there are other references to the parent
			 *  (i.e. this is not the last child) simply return
			 */
			if (--(P->refcount) > 0)
				return;

			if (dead(P))
			{	dealloc_ps(P);
				return;
			}

			/*
			 *  continue parent
			 */
			PS = P;
			Parent_PS = P->parent;
			P->refcount = 1;
			C = P->cont;
			FL = P->nextcl;
			Y = P->env;

			VERIFY_ps();
			break;


		case try:
			FL = (Code *) (WordP(C) + 1);
			C = (Code *) *WordP(C);
			break;

		case trust:
			FL = 0;
			C = (Code *) *WordP(C);
			break;

		case otherwise:
			/*  sequential clause separator  */
			if (Nsuspv > 0)
				goto Suspend_process;
			PS->cont = C;
			break;

		case otherwise_restore:
			if (PS->nextcl)
			{	/*
				 *  restore arguments after a mixed guard has
				 *  failed (suspended) in case the failure
				 *  (suspension) was in a flat guard test.
				 */
				copyargs(PS->args, Reg, PS->nargs);
			}

			/*  sequential clause separator  */
			if (Nsuspv > 0)
				goto Suspend_process;
			PS->cont = C;
			PS->nextcl = 0;
			break;

		Suspend:
			/*
			 *  suspend on the variable in t1: insert in
			 *  local suspension list if not already there
			 */
			for (j=0; j<Nsuspv; j++)
			{	if (VarTbl[j] == t1)
					goto Suspend2;
			}
			VarTbl[Nsuspv++] = t1;

		Suspend2:
			/*  try next clause  */
			if (FL)
			{	C = FL;
				break;
			}

			/*  no more alternatives, suspend process */
		Suspend_process:
			TRACE_suspend();

			if (PS->args == WNULL)
			{	PS->args = SP;
				SP += PS->nargs;
				copyargs(Reg, PS->args, PS->nargs);
				/* or...
				PS->args = SP;
				pushargs(Reg, PS->nargs);
				..... */

				if (PS->cont == 0)
					err(1, "suspend: no resumption addr\n");
			}

                        m_ht->m_top = HP;
                        m_st->m_top = SP;

			if (Nsuspv > 1)
				(void) multi_suspend(VarTbl, Nsuspv);
			else
				(void) suspend(ToPtr(*VarTbl));
			Nsuspv = 0;
			return;

/* S vjb 28/2/92	*/
		Suspend_for_event:

			if (PS->args == WNULL)
			{	PS->args = SP;
				SP += PS->nargs;
				copyargs(Reg, PS->args, PS->nargs);

				if (PS->cont == 0)
					err(1, "suspend_for_event: no resumption addr\n");
			}

			m_ht->m_top = HP;
			m_st->m_top = SP;

			return;
/* E vjb 28/2/92        */

		case begin_guard:
			/*
			 *  If alternative clause(s) exist save arguments
			 *  in case the guard fails.
			 */
			if (FL && PS->args == WNULL)
			{	PS->args = SP;
				SP += PS->nargs;
				copyargs(Reg, PS->args, PS->nargs);
			}

			PS->nextcl = FL;
			break;

		case end_guard:
			/*
			 *  check status of root process and then commit
			 */
			if (dead(PS->root))
				goto Fail;

			PS->nextcl = 0;
			/*  fall into...  */

		case commit:
			/*
			 *  reset FL, nsuspv and dealloc args on stack;
			 */
			FL = 0;
			Nsuspv = 0;
			P = PS;
			if (P->args != WNULL)
				dealloc_stack(P->args, P->nargs);
			P->nargs = 0;
			Ncalls++;		/*  a reduction!  */
			break;

		case fail:
		Fail:
			/*
			 *  if there's an alternative clause, try that;
			 *  else delete current process and fail parent
			 */
			if (PS->env != WNULL)
			{	Y = PS->env;
				/*  deallocate failed environment  */
				for (i=0; Y[i] != (Word) Y; i++)
					Y[i] = 0;

				Y[i++] = 0;
				if (SP == Y+i)
					SP = Y;
				PS->env = WNULL;
			}

			if (FL)
			{	C = FL;
				break;
			}

			if (Nsuspv > 0)
				goto Suspend_process;


		Deep_Fail:
			if (PS->args != WNULL)
				dealloc_stack(PS->args, PS->nargs);

			TRACE_fail();

			P = PS->parent;
			if (!P)			/*  top level fail  */
				err(1, "top level goal failed\n");

			/*
			 *  fail parent.
			 *  if parent has alternatives, try its next clause;
			 *  if there are other children then copy parent
			 *  to current PS and mark old parent as dead by
			 *  setting nargs to a negative number.
			 */
			i = P->nargs;
			refcnt = --(P->refcount);
			if (refcnt > 0)
				P->nargs = -1;

			if (i < 0)
			{	/*  parent has already failed, just return  */
				dealloc_ps(PS);
				if (refcnt == 0)
					dealloc_ps(P);

				m_ht->m_top = HP;
				m_st->m_top = SP;
				return;
			}

			if (refcnt > 0)
			{	*PS = *P;
				PS->nargs = i;		/* restore +ve nargs */
				PS->refcount = 1;

				/*  remove parent ref to args, env & nextcl  */
				P->args = WNULL;
				P->env = WNULL;
				P->nextcl = 0;
			}
			else
			{	dealloc_ps(PS);
				PS = P;
				PS->refcount = 1;
			}

			Parent_PS = PS->parent;
			FL = PS->nextcl;
			PS->nextcl = 0;

			if (FL)
				copyargs(PS->args, Reg, PS->nargs);

			goto Fail;


			/*
			 *  ENVIRONMENT INSTRUCTIONS
			 */
		case allocate:
			onereg(i);
			Y = PS->env = SP;
			SP += i;
			*SP++ = (Word) Y;
			break;

		case deallocate:
			for (i=0; Y[i] != (Word) Y; i++)
				Y[i] = 0;

			/*  remove top of env marker & check for top of stk  */
			Y[i++] = 0;
			if (SP == Y+i)
				SP = Y;

			PS->env = WNULL;
			break;


			/*
			 *	INDEXING INSTRUCTIONS
			 */
		case wait_switch_on_term:
			onereg(i);
			wait_for_argument(A[i]);

			i = u_tag(A[i]) - 1;
			C = (Code *) WordP(C)[i];

			/*  trust optimisation  */
			if (*C == trust)
			{	FL = 0;
				C = (Code *) *WordP(++C);
			}

			break;

		case switch_on_term:
			onereg(i);
			deref(A[i]);
			i = u_tag(A[i]);
			C = (Code *) WordP(C)[i];
			break;

			/*
			 *	BUILT IN PREDICATE CALLS
			 */
		case builtin_o:
			onereg(i);
			t1 = X[i];
			deref(t1);
			if (! (*BuiltinP(C))(t1) )
				goto Fail;

			C += WordArgSize;	/*  success  */
			break;

		case builtin_i:
			/*  wait for argument to be instantiated  */
			onereg(i);
			wait_for_argument(X[i]);

			if (! (*BuiltinP(C))(X[i]) )
				goto Fail;

			C += WordArgSize;	/*  success  */
			break;

		case builtin_io:
			/*  wait for 1st argument  */
			tworeg(i,j);
			wait_for_argument(X[i]);
			deref(X[j]);
			m_ht->m_top = HP;	/*  save SP? */
			if (! (*BuiltinP(C))(X[i], X[j]) )
				goto Fail;
			HP = m_ht->m_top;

			C += WordArgSize;	/*  success  */
			break;

		case builtin_ii:
			/*  wait for two arguments  */
			tworeg(i,j);
			wait_for_argument(X[i]);
			wait_for_argument(X[j]);

			if (! (*BuiltinP(C))(X[i], X[j]) )
				goto Fail;

			C += WordArgSize;	/*  success  */
			break;

		case builtin_nn:
			/*  wait for two numeric arguments  */
			tworeg(i,j);
			t1 = X[i];
			wait_for_argument(t1);
			t2 = X[j];
			wait_for_argument(t2);

			if (!IsNumber(t1&t2))
			{	if (IsNumber(t1))
					bu_arith_error(t2);
				else
					bu_arith_error(t1);
				goto Fail;
			}

			if (! (*BuiltinP(C))(t1,t2) )
				goto Fail;

			C += WordArgSize;	/*  success  */
			break;

		case builtin_ioo:
			/*  wait for first of 3 arguments  */
			onereg(i);
			t1 = X[i];
			wait_for_argument(t1);

			tworeg(i,j);
			deref(X[i]);
			deref(X[j]);
			m_ht->m_top = HP;
			if (! (*BuiltinP(C))(t1, X[i], X[j]) )
				goto Fail;
			HP = m_ht->m_top;

			C += WordArgSize;	/*  success  */
			break;

		case builtin_iio:
			/*  wait for first 2 of 3 arguments  */
			tworeg(i,j);
			t1 = X[i];
			wait_for_argument(t1);
			t2 = X[j];
			wait_for_argument(t2);

			onereg(i);
			deref(X[i]);
			m_ht->m_top = HP;
			if (! (*BuiltinP(C))(t1, t2, X[i]) )
				goto Fail;
			HP = m_ht->m_top;

			C += WordArgSize;	/*  success  */
			break;

			/*
			 *  FUNCTION CALLS
			 */

			/*
			 *  function instructions are used for is/2 calls and
			 *  are mainly in the body. Control transfers to the
			 *  label address unless an input argument is unbound
			 *  in which case the next instruction is executed.
			 */
		case function_n:
			onereg(i);
			t1 = X[i];
			deref(t1);
			if (IsNumber(t1))
			{	t1 = (*BuiltinP(C))(t1);
				C += WordArgSize;
				onereg(i);
				X[i] = t1;
				C = (Code *) *WordP(C);
				break;
			}

			C += (WordArgSize * 2) + 1;  /*  set to suspend code */
			if (!IsRef(t1))
			{	bu_arith_error(t1);
				goto Fail;
			}
			break;

		case function_nn:
			tworeg(i,j);
			t1 = X[i];
			deref(t1);
			t2 = X[j];
			deref(t2);

			if (IsNumber(t1&t2))
			{	t1 = (*BuiltinP(C))(t1,t2);
				C += WordArgSize;
				onereg(i);
				X[i] = t1;
				C = (Code *) *WordP(C);
				break;
			}

			C += (WordArgSize * 2) + 1;  /* set to suspend code */
			if (IsRef(t1))
				break;
			if (IsRef(t2))
			{	t1 = t2;
				break;
			}

			/*  must be error  */
			if (IsNumber(t1))
				bu_arith_error(t2);
			else
				bu_arith_error(t1);
			goto Fail;

		case increment:
			tworeg(i,j);
			t1 = X[i];
			deref(t1);

			if (IsShort(++t1))
			{	X[j] = t1;
				C = (Code *) *WordP(C);
			}
			else if (IsNumber(--t1))
			{	X[j] = make_int(int_val(t1)+1);
				C = (Code *) *WordP(C);
			}
			else if (IsRef(t1))
				C += WordArgSize;   /* set to suspend code */
			else
			{	bu_arith_error(t1);
				goto Fail;
			}
			break;

		case decrement:
			tworeg(i,j);
			t1 = X[i];
			deref(t1);

			if (IsShort(--t1))
			{	X[j] = t1;
				C = (Code *) *WordP(C);
			}
			else if (IsNumber(++t1))
			{	X[j] = make_int(int_val(t1)-1);
				C = (Code *) *WordP(C);
			}
			else if (IsRef(t1))
				C += WordArgSize;   /* set to suspend code */
			else
			{	bu_arith_error(t1);
				goto Fail;
			}
			break;

		case suspend_function:
			/*
			 *  used in guard only, immediately after function
			 *  instruction, which leaves unbound var in t1 (hack?)
			 */
			goto Suspend;

			/*
			 *	C-PREDICATE AND INTERPRETED CALLS
			 */
		case enter_c:
			m_ht->m_top = HP;	/* for copy_term+statistics */
			m_st->m_top = SP;	/* for copy_term+channels   */
			if ( (i = (*BuiltinP(C))(A)) != SUCCESS)
			{	if (i == SUSPEND)
					goto Suspend_process;
/* S vjb 28/2/92        */
				else if (i == SUSPEND_FOR_EVENT)
					goto Suspend_for_event;
/* E vjb 28/2/92        */
				else
					goto Deep_Fail;
			}
			SP = m_st->m_top;
			HP = m_ht->m_top;
			HE = m_ht->m_ovflw;

			C += WordArgSize;
			goto Proceed;

		case enter_io_c:
			if (ismaster(PR) && *io_q_front == PNULL)
			{	/*  master: executes I/O request directly  */
				m_ht->m_top = HP;
				i = (*BuiltinP(C))(A);
				if (i == SUCCESS)
				{	C += WordArgSize;
					HP = m_ht->m_top;
					goto Proceed;
				}
				else if (i == SUSPEND)
					goto Suspend_process;
				else if (i == FAIL)
					goto Deep_Fail;

				/*  else requeue process as below  */
			}

			/*
			 *  slave: queue the process
			 *  note: if any i/o request is outstanding
			 *  master will queue current process to
			 *  preserve order in which they are done -
			 *  this is not strictly necessary.
			 */
			if (PS->args == WNULL)
			{	PS->args = SP;
				SP += PS->nargs;
				copyargs(Reg, PS->args, PS->nargs);
			}
			PS->link = PNULL;

			lck = ptrtolck(io_q_front);
			lock(lck);
			if (*io_q_front != PNULL)
			{	(*io_q_back)->link = PS;
				*io_q_back = PS;
			}
			else
				*io_q_front = *io_q_back = PS;
			unlock(lck);

                        m_ht->m_top = HP;
                        m_st->m_top = SP;
			return;

		case enter_interpret:
			/*
			 *  given a structure in X[0] (and optionally a
			 *  module name in X[1]) locate procedure code
			 *  from functor and load arguments into registers
			 */
			t1 = A[0];
			wait_for_argument(t1);

			if (PS->nargs == 2)
			{	t2 = A[1];		/*  module_call(X,M) */
				wait_for_argument(t2);
				t2 = (Word) AtomVal(t2);
			}
			else
				t2 = 0;			/*  call(X)  */

			/*  deallocate argument(s) to "call"  */
			if (PS->args != WNULL)
			{	dealloc_stack(PS->args, PS->nargs);
				PS->args = WNULL;
			}

			if (IsAtom(t1))
			{	C = findproc(findfunct(AtomVal(t1), 0), (Atom *)t2)->p_code;
				PS->nargs = 0;
			}
			else if (IsStruct(t1))
			{	S = StructVal(t1);
				C = findproc(FunctVal(*S), (Atom *)t2)->p_code;

				i = FunctVal(*S)->f_arity;
				for (j=0; j<i; j++)
				{	S++;
					X[j] = IsUnb(*S) ? ToRef(S) : *S;
				}
				PS->nargs = i;
			}
			else
				C = proc_undefined->p_code;

			if (*C == enter_trace)
			{	/*  if $call(X), skip trace instruction  */
				if (WordP(PS->cont)[-1] == funct_call1)
					C += TraceOffset;
			}

			PS->cont = C;
			break;

		case enter_trace:
		case enter_undefined:
			/*
			 *  called procedure is being traced / is undefined.
			 *  build structure from functor name and args
			 *  and call $debug_goal/1 or $undefined_goal/1
			 */

			/*  functor name stored after trace/undef instr  */
			i = FunctVal(*WordP(C))->f_arity;
			if (i == 0)
				t1 = AsAtom(FunctVal(*WordP(C))->f_name);
			else
			{	t1 = AsStruct(HP);
				*HP++ = AsFunct(*WordP(C));

				for (j=0; j<i; j++)
					*HP++ = X[j];
			}
			X[0] = t1;

			if (C[-1] == enter_trace)
				C = proc_trace->p_code;
			else
				C = proc_undefined->p_code;

			if (PS->args != WNULL)
			{	dealloc_stack(PS->args, PS->nargs);
				PS->args = WNULL;
			}
			PS->nargs = 1;
			break;
		}
	}
}
