/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include <stdio.h>
#include <a.out.h>
#include <sys/file.h>
#include <strings.h>
#include "objs.h"
#include "macros.h"
#include "instr.h"
#include "ret.h"

#ifdef HERMES
#include <signal.h>
#include "hermes.h"
extern	int signal_mask;
int	status, old_mask;
#endif

#define MAXPREDS 64
#define MAXFILE	256

struct	nlist	nl_table[MAXPREDS];	/*  nlist table for load_foreign  */
struct	funct	*p_table[MAXPREDS];	/*  pred table for load_foreign  */

extern	Word	atom_nil;
extern	char	*basefile;


/*
 *  C_UNIX  --  $unix builtin predicate; performs some unix specific operation
 *		A0 contains operation number, A1 contains an (atom) argument.
 */
c_unix(Args)
Word	*Args;
{
	int	cmd;
	char	*cmdarg, *getenv();

	wait_for_argument(A0);
	cmd = ShortVal(A0);

	wait_for_argument(A1);
	if (!IsAtom(A1))
		return(FAIL);
	cmdarg = string_val(A1);

	switch (cmd)
	{	case 0:				/*  cd  */
			cmdarg = getenv("HOME");
		case 1:				/*  cd(directory)  */
			if (chdir(cmdarg) == 0)
				return(SUCCESS);
			break;

		case 2:				/*  shell  */
			cmdarg = getenv("SHELL");
		case 3:				/*  system(command)  */
#ifdef HERMES
			old_mask = sigblock(signal_mask);
			status = system(cmdarg);
			sigsetmask(old_mask);
			if (status == 0)
#else
			if (system(cmdarg) == 0)
#endif
				return(SUCCESS);
			break;

		case 4:				/*  access(file)  */
			if (access(cmdarg, R_OK) == 0)
				return(SUCCESS);
			break;
	}

	return(FAIL);
}


/*
 *  C_LOAD_FOREIGN  --  dynamically load in C predicates
 *			A0 contains list of structures specifying which C
 *			routines are predicates visible to Parlog;
 *			A1 contains the C object file (atom),
 *			A2 contains other files/libraries to be loaded (atom).
 *			A3 contains the enter instruction (int), must be
 *			either enter_c (85) or enter_io_c (86).
 *
 *			Note that arguments are assumed fully instantiated
 *			and are not type checked here.
 */
c_load_foreign(Args)
Word	*Args;
{
	static	char	strings[5120];
	static	char	cmd[256];
	Word	car, cdr;
	struct	funct	*f;
	int	i, npreds, enter;
	char	*strp = strings;
	char	*filearg, *ptr;
	char	objfile[MAXFILE], libs[MAXFILE];
	char	*mktemp();
	char	*valloc();
	unsigned alloc_fgn_code();
	char	loadfile[18];
	FILE	*fp;
	struct	exec	header;
	unsigned loadbase, lb2, loadsize;
	int	readsize;


	/*
	 *  collect list of predicates to be loaded and place them in
	 *  the predicate table and the nlist table.
	 *
	 *  note: each predicate is associated with C routine of the
	 *  same name prefixed by "C_";   e.g. test/2 --> C_test().
	 */
	npreds = 0;
	cdr = A0;
	deref(cdr);

	while (cdr != atom_nil)
	{	if (!IsList(cdr))
			break;

		car = *ListVal(cdr);
		deref(car);

		if (IsAtom(car))
			f = findfunct( AtomVal(car), 0 );
		else if (IsStruct(car))
			f = FunctVal( *StructVal(car) );
		else
			break;
		
		p_table[npreds] = f;
		nl_table[npreds].n_un.n_name = strp;
		(void) sprintf(strp, "_C_%s", f->f_name->a_string);
		strp += f->f_name->a_length + 4;

		if (++npreds > MAXPREDS)
			break;

		cdr = *(ListVal(cdr) + 1);
		deref(cdr);
	}

	if (cdr != atom_nil)
	{	bu_error(A0, "load_foreign: 1st argument incorrect");
		return(FAIL);
	}

	nl_table[npreds].n_un.n_name = "";	/*  mark end of nlist  */

#ifdef DEBUG
	printf("load_foriegn: %d predicates\n", npreds);
#endif


	deref(A1);
	filearg = string_val(A1);
	if (!ic_file_name(filearg, objfile, TRUE))
	{	bu_error(make_atom(filearg), "load_foreign: cannot open");
		return(FAIL);
	}
	deref(A2);
	filearg = string_val(A2);
	fprintf(user_error, "{loading foreign from %s %s}\n", objfile, filearg);
	*libs = '\0';
	if (strcmp(filearg, " "))
	{	*strp = ' ';
		while (ptr = index(filearg, ' '))
		{	*ptr++ = '\0';
			if (*filearg == '-')
			{	(void) strcat(libs, " ");
				(void) strcat(libs, filearg);
			}
			else
			{
				if (!ic_file_name(filearg, strp+1, TRUE))
				{	bu_error(make_atom(filearg), "load_foreign: cannot open");
					return(FAIL);
				}
				(void) strcat(libs, strp);
			}
			filearg = ptr;
		}
		if (*filearg == '-')
		{	(void) strcat(libs, " ");
			(void) strcat(libs, filearg);
		}
		else
		{
			if (!ic_file_name(filearg, strp+1, TRUE))
			{	bu_error(make_atom(filearg), "load_foreign: cannot open");
				return(FAIL);
			}
			(void) strcat(libs, strp);
		}
	}
	deref(A3);
	enter = ShortVal(A3);
	if (enter != enter_c) enter = enter_io_c;	/*  safety  */

	/*
	 *  set up the ld command.
	 *
	 *  we grab a page of memory in the hope that this is where the code
	 *  will go and give that as the loadbase.  Once the ld is done the
	 *  real space required is known and we allocate that.  If the real
	 *  loadbase turns out to be different then the ld is repeated with
	 *  the new value for loadbase.
	 *
	 *  Note that the routines are (currently) loaded in to private
	 *  memory; so all such predicates must be run by the same processor
	 *  as is doing this load - i.e. the master
	 */
	(void) strcpy(loadfile, "/tmp/parldXXXXXX");
	(void) mktemp(loadfile);

	loadsize = (unsigned) getpagesize();
	if (enter == enter_c)
		loadbase = alloc_fgn_code(loadsize);
	else
		loadbase = (unsigned) valloc(loadsize);

again:	(void) sprintf(cmd, "ld -N -x -A %s -T %x -o %s %s %s -lc",
			     basefile,  loadbase, loadfile, objfile, libs);

#ifdef DEBUG
	printf("load_foreign: cmd = %s\n", cmd);
#endif

#ifdef HERMES
	old_mask = sigblock(signal_mask);
	status = system(cmd);
	sigsetmask(old_mask);
	if (status)
#else
	if (system(cmd))
#endif
	{	bu_error(A1, "load_foreign: ld command failed");
		return(FAIL);
	}


	/*
	 *  read the header of the loadfile produced by ld;
	 *  allocate space for it and read the text+data in.
	 */
	
	if ((fp = fopen(loadfile, "r")) == NULL)
	{	bu_error(make_atom(loadfile), "load_foreign: cannot open");
		return(FAIL);
	}

	if (fread((char *)&header, sizeof(header), 1, fp) != 1)
	{	bu_error(make_atom(loadfile), "load_foreign: cannot read");
		(void) fclose(fp);
		return(FAIL);
	}

	readsize = header.a_text + header.a_data;
	if ((readsize + header.a_bss) > loadsize)
	{	loadsize = readsize + header.a_bss;

		if (enter == enter_c)		/* load into shared mem  */
		{	free_fgn_code(loadbase);
			lb2 = alloc_fgn_code(loadsize);
		}
		else				/* load into private mem */
		{	free((char *) loadbase);
			if ((lb2 = (unsigned) valloc(loadsize)) == NULL)
			{	bu_error(A1, "load_foreign: valloc failed");
				return(FAIL);
			}
		}

		if (lb2 != loadbase)
		{	/*  repeat ld with corrected loadbase  */
			(void) fclose(fp);
			loadbase = lb2;
			goto again;
		}
	}

	if (fread((char *)loadbase, 1, readsize, fp) != readsize)
	{	bu_error(make_atom(loadfile), "load_foreign: wrong length");
		(void) fclose(fp);
		return(FAIL);
	}

	(void) fclose(fp);

	/*  get entry points into the C functions and remove load file  */
	if (nlist(loadfile, nl_table) == -1)
	{	bu_error(make_atom(loadfile), "load_foreign: no namelist");
		return(FAIL);
	}
	(void) unlink(loadfile);

	/*  now define each C predicate that's in the predicate table  */
	for (i=0; i < npreds; i++)
	{	f = p_table[i];

		if (nl_table[i].n_value == 0)
		{	bu_error(AsFunct(f), "load_foreign: C function undefined");
			continue;
		}

		define_c_predicate(f->f_name->a_string, f->f_arity,
				   (int (*) ()) nl_table[i].n_value,
				   enter, 0);
	}

	return(SUCCESS);
}
