/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include <stdio.h>
#include <signal.h>
#include <sys/wait.h>
#include "objs.h"
#include "proc.h"
#include "mem.h"
#include "synch.h"
#include "event.h"
#include "instr.h"
#include "ret.h"
#include "trace.h"
#include "timer.h"

extern	FILE	*open_ic_file();

#ifdef HERMES
#include "stream.h"
#include "hermes.h"
void	hermes_top();
extern	void	timeslice();
extern	bool	h_debug;
extern	int	wait_pipe;
extern	int	h_deadlock;
extern	threadpo prolog_th;
FILE	*user_error = stderr;
int	tty_pid;
int	x_parlog = 0;
#endif
#define HEAPSIZE	(2 + 4/NPR)	/*  initial heap segments    */
#define	NSEGMENTS	(5 * NPR) + 7	/*  (HEAPSIZE+3)*NPR + 3     */

#define CODESIZE	5		/*  codesize if private code */

/*
 *  shared memory and data area pointers
 */
Mem	*shmem_base;			/*  shared memory area	*/
char	*lock_base;			/*  lock area		*/

Mem	**mem_free;			/*  global free memory ptr   */
int	*mem_nfree;			/*  no. global free segments */
Mem	*m_free;			/*  local mem freelist ptr   */
int	m_nfree;			/*  no. local free segments  */

int	nsegments;			/*  no. segments to allocate */

Mem	*m_hb, *m_sb, *m_pb;		/*  ptrs to bottom segments  */
Mem	*m_ht, *m_st, *m_pt;		/*  ptrs to top segments     */

Code	**code_top,**code_end;		/*  code pointers	*/
int	*code_size;			/*  code size		*/

int	*sys_base;			/*  system area		*/



/*
 *  processor variables
 */
Processor *PR;				/*  this processor	*/
Process	*PS;				/*  current process	*/
Process *Parent_PS;			/*  current parent	*/


Process	*p_top, *p_end;			/*  process stack ptrs	*/
Process	*p_free;			/*  process freelist	*/
Process	**q_back;			/*  local q back ptr	*/
Code	*c_top, *c_end;			/*  code pointers	*/

Processor *Pr0, *PrN;			/*  1st & last pr's	*/
int	NPR;				/*  no. of processors	*/

int	Ncalls;				/*  no. of goal calls	*/
int	Nsuspensions;			/*  no. of goal susps	*/


/*
 *  shared memory variables
 */
int	*bars;				/*  ptr to barriers	*/
int	*event;				/*  shared event flag   */

int	*gc_counter;			/*  g.c. counter	*/

Process	**io_q_front;			/*  front of I/O queue	*/
Process	**io_q_back;			/*  back of I/O queue	*/

Word	**intrpt_var;			/*  interrupt variable  */

int	*st_ngcs;			/*  no. of g.c's	*/
int	*st_utime;			/*  user cpu time 	*/
int	*st_last_utime;			/*  last user cpu time  */

/*
 *  symbol table variables
 */
Atom	**a_htable;			/*  atom hash table	*/
Number	**n_htable;			/*  number hash table	*/
Atom	**a_curr_module;		/*  current module name */



/*
 *  MAIN  --  the main program
 */
#ifdef HERMES
jam(argc,argv)
#else
main(argc,argv)
#endif
int	argc;
char	*argv[];
{
	int	val, ch;
	char	*ap;
	char	*objectfile;
	FILE	*fp;
	char	*init_shmem();
	char	*init_locks();
	SIG_T	catch_intrpt(), catch_alarm(), catch_bad();
#ifdef sun
	SIG_T	catch_quit();
#endif
	int	syssize, memsize;
	int	sys_segs, i;
	union	wait status;

#ifdef HERMES
	/* stop timeslicing while initialising, remember
	   to turn it back on (in Parlog) when done */
	timeslice(0);
#endif
	/*
	 *  process the command line arguments
	 */
	NPR = 1;
	nsegments = 0;


	objectfile = argv[1];
	argv++;
	argc--;

	while (argc > 1 && argv[1][0] == '-')
	{	ap = &argv[1][1];
		switch( *ap )
		{	/*  ordinary options  */
#ifdef TRACE
			case 'h':
				instructions[halt].i_trace = 0;
				break;

			case 'v':
				verifyflag = 1;
				break;
#endif TRACE
			default:
				/*  options with numeric argument  */
				if (ap[1])
					val = atoi(&ap[1]);
				else if (argc > 2)
				{	argc--;
					argv++;
					val = atoi(argv[1]);
				}
				else val = 0;

				switch( *ap )
				{	case 'm':
						nsegments = val;
						break;
#ifdef TRACE
					case 't':
						traceflag = val;
						break;
#endif TRACE
					default:
						err(-1, "unknown flag: %c\n", *ap);
				}
		}
		argc--;
		argv++;
	}

	if (argc != 1)
		err(-1, "parlog: invalid start up arguments\n");

	if (nsegments == 0)
		nsegments = NSEGMENTS;


	/*
	 *  allocate shared memory and locks
	 */
	syssize = NPR * sizeof(Processor)
		+ H_TBLSIZ * sizeof(Atom *)
		+ H_TBLSIZ * sizeof(Number *)
		+ NBARRIERS * sizeof(int)
		+ 16 * sizeof(int);

	sys_segs = (syssize / sizeof(Mem)) + 1;
	memsize = (sys_segs + nsegments) * sizeof(Mem);

	shmem_base = (Mem *) init_shmem(memsize, sizeof(Mem));
	lock_base  = init_locks(NLOCKS);

	Pr0 = (Processor *) shmem_base;
	PrN = Pr0 + NPR;

	a_htable = (Atom **) PrN;
	n_htable = (Number **) (a_htable + H_TBLSIZ);

	sys_base = (int *) (n_htable + H_TBLSIZ);

	/*
	 *  initialise barriers and other shared variables
	 */
	bars = sys_base;
	sys_base += NBARRIERS;
	for (i=0; i < NBARRIERS; i++)
		bars[i] = 0;

	mem_free   = (Mem **)	  sys_base++;
	mem_nfree  = (int *)	  sys_base++;
	code_top   = (Code **)	  sys_base++;
	code_end   = (Code **)	  sys_base++;
	code_size  = (int *)	  sys_base++;
	event	   = (int *)	  sys_base++;
	gc_counter = (int *)	  sys_base++;
	io_q_front = (Process **) sys_base++;
	io_q_back  = (Process **) sys_base++;
	intrpt_var = (Word **)	  sys_base++;
	st_ngcs	   = (int *)	  sys_base++;
	st_utime   = (int *)	  sys_base++;
	st_last_utime = (int *)	  sys_base++;
	a_curr_module = (Atom **) sys_base++;


	/*
	 *  remainder of sys segment is used for initial code area
	 */
	*code_top = (Code *) sys_base;
	*code_end = (Code *) (shmem_base + sys_segs);
	*code_size = 0;

	*mem_free = shmem_base + sys_segs;
	*mem_nfree = nsegments;

	*io_q_front = PNULL;
	*intrpt_var = WNULL;
	*st_ngcs = 0;
	*st_utime = 0;
	*st_last_utime = 0;

	configure(0);

	/*
	 *  initialise predefined names and builtins
	 */
	initialise();


	/*
	 *  open the boot file;  skip the 1st line if its an interpreter line,
	 *  i.e. of the form "#!/xxxx/jam\n", and then load the rest.
	 */
	if ((fp = open_ic_file(objectfile, "r", 1)) == NULL)
		err(-1,"cannot open boot file \"%s\"\n", objectfile);

        if ((ch = getc(fp)) == '#')
        {       while ((ch = getc(fp)) != '\n' && ch != EOF)
                        continue;
        }
        else
                (void) ungetc(ch, fp);

	if (load(fp) == FAIL)
		err(-1,"failed to load boot file \"%s\"\n", objectfile);


	/*
	 *  initialise signals
	 */
#ifndef HERMES
	(void) signal(SIGINT,  catch_intrpt);	/*  in signal.c  */
#endif
	(void) signal(SIGALRM, catch_alarm);	/*  in sched.c   */
#ifdef sun
	(void) signal(SIGQUIT, catch_quit);	/*  in signal.c  */
#endif
/*	(void) signal(SIGSEGV, catch_bad);	/*  in signal.c  */
	(void) signal(SIGBUS,  catch_bad);	/*  in signal.c  */

	/*  initialise microsecond timer  */
	init_clock();
	 
	/*
	 *  create child processes to act as other processors
	 */
	for (i=1; i < NPR; i++)
	{	if (fork() == 0)
		{	configure(i);
			top();
			icp_exit(0);
		}
	}

	/*
	 *  parent executes as master processor
	 */
#ifdef HERMES
	wait_pipe = 0;
	barrier(bars[1], NPR);
	return(x_parlog ? SUCCESS : SUSPEND);
}

threadpo
jam_end()
{
	int	i;
	union	wait status;
#else
	top();
#endif
	/*
	 *  cleanup child processes. check all exited nicely
	 */
	for (i=1; i < NPR; i++)
	{	if (wait(&status) > 0)
		{	if (status.w_status > 0)
			{	printf("warning: a child process died");
				if (status.w_retcode)
					printf(", status %d", status.w_retcode);
				if (status.w_termsig)
					printf(", signal %d", status.w_termsig);
				printf("\n");
			}
		}
	}

#ifdef HERMES
	if (x_parlog) {
		fclose(st_user_input->st_fp);
		fclose(st_user_output->st_fp);
		kill(tty_pid, SIGTERM);	/* kill off tty, if any */
	}
	else {
		fprintf(stderr, "\n[ switching to Prolog console ]\n");
		(void) add_to_runq(prolog_th, FALSE);
	}

	/* exit from parlog, so unlink thread */
	(void) remove_from_runq(parlog_th);
	parlog_th->prev->next = parlog_th->next;
	parlog_th->next->prev = parlog_th->prev;
	free((char *)parlog_th);
	free((char *)shmem_base);
	free((char *)lock_base);
	parlog_th = NULL;
#else
	exit(0);
#endif
}


/*
 *  CONFIGURE  --  initialise processor data area pointers
 */
configure(n)
int	n;
{
	int	nsegs;

	PR = &Pr0[n];

	PR->q_bot = PR->pr_q;
	PR->q_end = PR->pr_q + sizeof(PR->pr_q) / sizeof(Process *);
	PR->q_front = PR->q_back = q_back = PR->q_bot;

	/*
	 *  Create processor memory freelist. HEAPSIZE segments are then
	 *  allocated for the heap and one for each stack, leaving one in the
	 *  local free list. The rest (3 by default) go in the global pool.
	 *  HEAPSIZE is 2 for NPR > 4, rising to 6 for NPR == 1.
	 */
	nsegs = HEAPSIZE + 3;
	m_free = get_new_segments(nsegs);
	m_nfree = nsegs;

	h_gen = s_gen = 0;

	m_pb = m_pt = alloc_new_proc_segment();
	m_sb = m_st = alloc_new_stack_segment();
	m_hb = m_ht = alloc_new_heap_segment();
	(void) request_heap_segments(HEAPSIZE - 1);

	p_top = (Process *) m_pt->m_data;
	p_end = (Process *) m_pt->m_ovflw;
	m_pt->m_top = (Word *) p_end;
	p_free = PNULL;

	c_top = c_end = (Code *) 0;

	PR->pr_state = PR_EXECUTE;
	PR->pr_lock = 0;
}


/*
 *  TOP  --  top level processor execution.
 *
 *	Wait at the barrier until all processors have initialised.
 *	Then go into a search-for-work/execute loop until a
 *	"halt" instruction is executed (or deadlock/failure occurs).
 */
top()
{
	barrier(bars[1], NPR);

	TRACE_msg("starts");

	while ( search_for_work() )
		execute();

	TRACE_msg("exits");

#ifdef CNTS
#define C(n)	PR->st_cnts[n]
	printf("counters: %3d %3d %3d  %3d %3d %3d\n",
		C(0), C(1), C(2), C(3), C(4), C(5));
#undef C
#endif
}

#ifdef HERMES
void
hermes_top(th)
threadpo th;
{	int	old_Ncalls;
	SIG_T	catch_intrpt();

	TH = th;
	old_Ncalls = Ncalls;

	if (h_debug > 1)
		fprintf(stderr, "[ resume parlog ]\n");

	while ( search_for_work() )
		execute();

	if (h_debug > 1)
		fprintf(stderr, "[ suspend parlog ]\n");

	if (Ncalls > old_Ncalls)
		h_deadlock &= ~(D_PROLOG | D_DEADLOCK);

	if (!TH) {

#ifdef CNTS
#define C(n)	PR->st_cnts[n]
		printf("counters: %3d %3d %3d  %3d %3d %3d\n",
			C(0), C(1), C(2), C(3), C(4), C(5));
#undef C
#endif
		jam_end();
	}
}
#endif

/*
 *  ERR  --  print an error message on diagnostic output
 *	     If error is fatal then set termination event and exit.
 */
/*VARARGS2*/
err(fatal, fmt, arg)
int	fatal;
char	*fmt, *arg;
{
#ifdef HERMES
	if (fatal)
	{
		fprintf(stderr, "Parlog error: ");
		fprintf(stderr, fmt, arg);
		if (fatal > 0)
		{	*event = E_TERM;	/*  set termination event  */
			(void) execute_event();	/*  go there for barrier   */
		}

		if (x_parlog) {
			fclose(st_user_input->st_fp);
			fclose(st_user_output->st_fp);
			kill(tty_pid, SIGTERM);
		}
		icp_exit(fatal);
	}
	fprintf(user_error, "Parlog warning: ");
	fprintf(user_error, fmt, arg);
#else
	fprintf(stderr, "%s: ", fatal ? "error" : "warning");
	fprintf(stderr, fmt, arg);

	if (fatal)
	{	if (fatal > 0)
		{	*event = E_TERM;	/*  set termination event  */
			(void) execute_event();	/*  go there for barrier   */
		}

		exit(fatal);
	}
#endif
}
