/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include <stdio.h>
#include <ctype.h>
#include "objs.h"
#include "proc.h"
#include "mem.h"
#include "macros.h"
#include "synch.h"
#include "instr.h"
#include "event.h"
#include "ret.h"

/*
 *  load codes
 */
#define	LD_S_NUM	'a'
#define	LD_L_NUM	'b'
#define	LD_F_NUM	'c'
#define	LD_ATOM		'd'
#define	LD_FUNCT	'e'
#define	LD_PROC		'f'
#define	LD_LABEL	'g'
#define LD_BUILTIN	'h'
#define	LD_CODE		'i'
#define	LD_CMD		'j'
#define	LD_END		'k'


/*
 *  object sizes (in code-items)
 */
#define	numbersize	sizeof(struct number) / sizeof(Code)
#define	atomsize(x)	(sizeof(struct atom) + x) / sizeof(Code) + 1
#define	functsize	sizeof(struct funct)  / sizeof(Code)
#define	procsize	sizeof(struct proc)   / sizeof(Code)


Word	ld_symtbl[1024];		/*  load symbol table	*/

extern	Code	**code_top, **code_end;
extern	Word	funct_cmd;
extern	Proc	*proc_cmd;
extern	Proc	*proc_load;
extern	Atom	**a_curr_module;
extern	Mem	*shmem_base;

extern	FILE	*open_ic_file();

#define PCodeOffset	0
#define	alloc_shd_code(n)	alloc_code(n)



/*
 *  MODULE PREDICATES
 */
c_module(Args)
Word	*Args;
{
	wait_for_argument(A0);
	if (!IsAtom(A0))
	{	bu_error(A0, "module/1: invalid argument");
		return(FAIL);
	}

	*a_curr_module = AtomVal(A0);
	return(SUCCESS);
}

c_curr_module(Args)
Word	*Args;
{
	return( unify(A0, AsAtom(*a_curr_module)) );
}

c_public(Args)
Word	*Args;
{
	/*  set the procedure's public flag  */
	return( set_proc_flag(A0, P_PUBLIC) );
}

c_export(Args)
Word	*Args;
{
	/*  set the procedure's export flag  */
	return( set_proc_flag(A0, P_EXPORT) );
}

c_import(Args)
Word	*Args;
{
	Atom	*imp_module;
	Funct	*f;
	Proc	*sp, *cp;

	wait_for_argument(A0);
	imp_module = AtomVal(A0);

	wait_for_argument(A1);
	if (IsAtom(A1))
		f = findfunct( AtomVal(A1), 0 );
	else if (IsStruct(A1))
		f = FunctVal( *StructVal(A1) );
	else
		return(FAIL);

	cp = findproc(f, ANULL);
	if (cp->p_flags & P_READONLY)
		return(FAIL);

	sp = findproc(f, imp_module);
	if (cp == sp)
		return(SUCCESS);

	if (!(sp->p_flags & P_EXPORT))
		return(FAIL);

	cp->p_code = sp->p_code;
	return(SUCCESS);
}


/*
 *  LOAD PREDICATES --  builtin predicates to interface to load()
 */
c_load(Args)
Word	*Args;
{
	char	*filename;
	FILE	*fp;

	wait_for_argument(A0);
	if (!IsAtom(A0))
	{	bu_error(A0, "load/1: invalid argument");
		return(FAIL);
	}

	filename = string_val(A0);
	if ((fp = open_ic_file(filename, "r", 1)) == NULL)
		return(FAIL);

	return( load(fp) );
}

c_load1(Args)
Word	*Args;
{
	/*  A0 contains file descriptor as a short integer  */
	return( load((FILE *)int_val(A0)) );
}


/*
 *  LOAD  --  load an object file into code space
 */
load(fp)
FILE	*fp;
{
	int	i, ld_key;
	Atom	*a;
	Funct	*f;
	Proc	*p;
	Code	*codep;
	char	*get_string();
	Proc	dummyp;

	while ((ld_key = getc(fp)) != EOF)
	{	switch(ld_key)
		{
		case LD_ATOM:
			i = get_int(fp);
			ld_symtbl[i] = make_atom( get_string(fp) );
			break;

		case LD_FUNCT:
			i = get_int(fp);
			a = AtomVal( ld_symtbl[ get_int(fp) ] );
			ld_symtbl[i] = AsFunct( findfunct( a, get_int(fp) ) );
			break;

		case LD_PROC:
			i = get_int(fp);
			f = FunctVal( ld_symtbl[ get_int(fp) ] );
			ld_symtbl[i] = (Word) findproc(f, ANULL);

			break;

		case LD_CODE:
			i = get_int(fp);
			p = (Proc *) ld_symtbl[3];

			if (p->p_module != *a_curr_module ||
			    (p->p_flags & P_READONLY))
			{	bu_error(ld_symtbl[2], "can't redefine predicate");
				p = &dummyp;	/* allows load to continue */
				p->p_size = 0;
			}

			/*  overwrite existing code if possible  */
			if (i <= p->p_size)
				codep = p->p_code - TraceOffset + PCodeOffset;
			else
			{	codep = alloc_shd_code(i+3);
				p->p_code = codep + TraceOffset - PCodeOffset;
				p->p_size = i;
			}
#ifdef ALIGN
			bzero((char *) codep, (i+3) * sizeof(Code));
#endif
			get_bytecode(fp, codep, ld_symtbl[2]);
			break;

		case LD_CMD:
			i = get_int(fp);
			codep  = alloc_shd_code(i+3);
			proc_cmd->p_code = codep + TraceOffset;
			proc_cmd->p_size = i;
#ifdef ALIGN
			bzero((char *) codep, (i+3) * sizeof(Code));
#endif
			get_bytecode(fp, codep, funct_cmd);

			/*  see if at end of file  */
			ld_key = getc(fp);
			if (ld_key == LD_END)
			{	(void) fclose(fp);
				load_command(fp, 1);
			}
			else
			{	(void) ungetc(ld_key, fp);
				load_command(fp, 0);
			}
			return(SUCCESS);

		case LD_END:
			(void) fclose(fp);
			return(SUCCESS);
		}
	}

	/*  premature eof  */
	(void) fclose(fp);
	return(FAIL);
}


/*
 *  LOAD_COMMAND  --  setup a process with a code sequence to call proc_cmd
 */
load_command(fp, atend)
FILE	*fp;
int	atend;
{
	register Code	*cp;
	register Process *ps;
	Code	*codep;
	char	*lck;

	/*
	 *  setup code sequence - if not at end arrange to call proc_cmd
	 *  and load again; otherwise just call proc_cmd
	 */
	if (atend)
	{	codep = cp = alloc_shd_code(4);
		*cp++ = call_promoted_last;
		*cp++ = 0;
		*ProcP(cp) = proc_cmd;
	}
	else
	{	codep = cp = alloc_shd_code(12);
		*cp++ = call_last;
		*cp++ = 0;
		*ProcP(cp) = proc_cmd;
		cp += WordArgSize;
		*cp++ = put_constant;		/*  store "fp" as an int  */
		*cp++ = 0;
		*WordP(cp) = make_int((int)fp);
		cp += WordArgSize;
		*cp++ = call_promoted_last;
		*cp++ = 1;
		*ProcP(cp) = proc_load;
	}

	/*
	 *  create process to execute command. If called from c_load(), add it
	 *  to process tree as a sibling to $load call. Then add to run queue.
	 */
	alloc_ps(ps);
	ps->parent = PNULL;
	ps->cont = codep;
	ps->nextcl = 0;
	ps->args = WNULL;
	ps->nargs = 0;
	ps->root = PNULL;
	ps->refcount = 1;

	if (PS && PS->parent)
	{	ps->parent = PS->parent;
		if (PS->parent->refcount == 1)
			PS->parent->refcount++;
		else
		{	lck = ptrtolck(PS->parent);
			lock(lck);
			PS->parent->refcount++;
			unlock(lck);
		}
	}

	enqueue_process(ps);
}


/*
 *  functions to read tokens from load file
 */
char	tmpword[A_STRSIZ];
#define	readword(f)	{ char *wp; for (wp=tmpword; (*wp = getc(f)); wp++); }

get_int(fp)
FILE	*fp;
{
	readword(fp);
	return( atoi(tmpword) );
}

double
get_float(fp)
FILE	*fp;
{
	double atof();

	readword(fp);
	return( atof(tmpword) );
}

char	*
get_string(fp)
FILE	*fp;
{
	readword(fp);
	return( tmpword );
}


get_bytecode(fp,cp,fn)
FILE	*fp;
register Code	*cp;
Word	fn;
{
	int	c, i;
	double	f, get_float();
	union	int_float num;
	Code	*proc_start;
	Number	*n;
	char	*wp;
	extern	Word	(*bu_table[])();
#ifdef ALIGN
	Code	*lcp = cp + TraceOffset;
	Code	*lp, *lp1;
#endif

	/*  store procedure header  */
	*cp++ = enter_trace;
	*WordP(cp) = fn;
	cp +=  WordArgSize;
	proc_start = cp;

	while ((c = getc(fp)) > 0)
	{
#ifdef ALIGN
		if (*lcp)
		{	/*  realign pointers (labels) pointing to here  */
			lp = (Code *) *lcp;
			while (lp)
			{	lp1 = lp;
				lp = (Code *) *lp;

				/* make labels point to private space */
				*lp1 = (Code) (cp - PCodeOffset);
			}
		}

		lcp++;
		if (c >= LD_S_NUM)
			lcp++;
#endif

		switch(c)
		{
		case LD_S_NUM:
			i = get_int(fp);
			*WordP(cp) = ToShort(i);
			cp += WordArgSize;
			break;

		case LD_L_NUM:
			i = get_int(fp);
			num.N_int = i;
			n = findnumber(N_INT, num);
			*WordP(cp) = AsNumber(n);
			cp += WordArgSize;
			break;

		case LD_F_NUM:
			f = get_float(fp);
			num.N_float = f;
			n = findnumber(N_FLOAT, num);
			*WordP(cp) = AsNumber(n);
			cp += WordArgSize;
			break;

		case LD_ATOM:
		case LD_FUNCT:
		case LD_PROC:
			i = get_int(fp);
			*WordP(cp) = ld_symtbl[i];
			cp += WordArgSize;
			break;

		case LD_LABEL:
			i = get_int(fp);
#ifdef ALIGN
			/*  add cp to realignment chain  */
			*cp = proc_start[i];
			proc_start[i] = (Code) cp++;
#else
			*WordP(cp) = (Word) (proc_start + i - PCodeOffset);
			cp += WordArgSize;
#endif
			break;

		case LD_BUILTIN:
			i = get_int(fp);
			*WordP(cp) = (Word) bu_table[i];
			cp += WordArgSize;
			break;

		default:
			if (c < '0' || c > '9')
				err(1, "bad symbol in get_bytecode %c\n", c);

			tmpword[0] = c;
			for (wp = &tmpword[1]; (*wp = getc(fp)); wp++)
				;
			*cp++ = atoi(tmpword);
			break;
		}
	}
}


/*
 *  symbol table lookup and insert functions
 */

Number	*
findnumber(type,num)
int	type;
union	int_float  num;
{
	Number	*n, *np;
	int	h;
	char	*lck;

	/*  get hash key  */
	h = num.N_int & (H_TBLSIZ - 1);
	
again:	np = NULL;
	n = n_htable[h];

	while (n)
	{	if (n->n_type == type)
		{	if (type == N_INT && n->n_int == num.N_int)
				return(n);
			if (type == N_FLOAT && n->n_float == num.N_float)
				return(n);
		}

		np = n;
		n = n->n_next;
	}

	/*  make new number  */
#if (defined(sparc) && defined(__GNUC__))
	if (type == N_FLOAT)
	{	n = (Number *) alloc_code( numbersize + 1 );
		if ((int) &n->n_num.N_float & 0x4)
			n = (Number *) ((char *)n + 4);
	}
	else
#endif
	n = (Number *) alloc_code( numbersize );
#if defined(__GNUC__)
	if ((n->n_type = type) == N_FLOAT)
		n->n_num  = num;
	else
		n->n_num.N_int = num.N_int;
#else
	n->n_type = type;
	n->n_num  = num;
#endif
	n->n_next = NULL;

	/*  insert into symbol table  */
	lck = ptrtolck(n_htable);
	if (np)
	{	lock(lck);
		if (np->n_next == NULL)
			np->n_next = n;
		unlock(lck);

		if (np->n_next != n)
			goto again;
	}
	else
	{	lock(lck);
		if (n_htable[h] == NULL)
			n_htable[h] = n;
		unlock(lck);

		if (n_htable[h] != n)
			goto again;
	}

	return(n);
}


/*
 *  character types in atoms
 *	1 = a..z
 *	2 = A..Z, 0..9, _
 *	4 = #$&*+-./:<=>?@\^`~
 *	8 = other characters
 */
int	char_tab[128] = 
{	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 4, 4, 8, 4,16, 8, 8, 4, 4, 8, 4, 4, 4,
	2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 4, 8, 4, 4, 4, 4,
	4, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
	2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 8, 4, 8, 4, 2,
	4, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 8, 8, 8, 4, 8
};

Atom	*
findatom(s)
char	*s;
{
	Atom	*a, *ap;
	int	i, h, len, prp;
	register char	*s1, *p, *q;
	char	*lck;

	/*  get hash key  */
	h = 0;
	for (s1=s; *s1; s1++)
		h += *s1;
	h &= H_TBLSIZ - 1;

again:	ap = NULL;
	a = a_htable[h];

	/*  find atom  */
	len = s1 - s;
	while (a)
	{	if (len == a->a_length)
		{	p = a->a_string;
			q = s;
			for (i=0; i<len; i++)
				if (*p++ != *q++)
					break;
			if (i == len)
				return(a);
		}

		ap = a;
		a = a->a_next;
	}

	/*
	 *  determine quoting properties: an atom needs quoting unless
	 *  (a) it contains a lc letter followed only by alphanumerics and '_'
	 *  (b) it contains only special characters.
	 */
	prp = 0;
	for (s1=s, i=0; *s1; s1++)
		i |= char_tab[*s1];
	
	if ((char_tab[*s] != 1 && i != 4) || i > 4 || (len == 1 && *s == '.'))
		prp = A_QUOTE;		/*  atom needs quotes  */
	if (i >= 16)
		prp |= A_ESCAPE;	/*  atom contains a ' */

	/*  make new atom  */
	a = (Atom *) alloc_code( atomsize(len) );
	a->a_next = NULL;
	a->a_funct = NULL;
	a->a_ops = NULL;
	a->a_channel = PNULL;
	a->a_props = prp;
	a->a_length = len;
	bcopy(s, a->a_string, len);
	a->a_string[len] = '\0';

	/*  insert into symbol table  */
	lck = ptrtolck(a_htable);
	if (ap)
	{	lock(lck);
		if (ap->a_next == NULL)
			ap->a_next = a;
		unlock(lck);

		if (ap->a_next != a)
			goto again;
	}
	else
	{	lock(lck);
		if (a_htable[h] == NULL)
			a_htable[h] = a;
		unlock(lck);

		if (a_htable[h] != a)
			goto again;
	}

	return(a);
}

Funct	*
findfunct(a,i)
Atom	*a;
int	i;
{
	Funct	*f, *fp;
	char	*lck;

again:	fp = NULL;
	f = a->a_funct;

	/*  find functor  */
	while (f)
	{	if (f->f_arity == i)
			return(f);

		fp = f;
		f = f->f_next;
	}

	/*  make new functor  */
	f = (Funct *) alloc_code( functsize );
	f->f_name = a;
	f->f_arity = i;
	f->f_next = NULL;
	f->f_proc = NULL;

	/*  insert into symbol table  */
	lck = ptrtolck(a_htable);
	if (fp)
	{	lock(lck);
		if (fp->f_next == NULL)
			fp->f_next = f;
		unlock(lck);

		if (fp->f_next != f)
			goto again;
	}
	else
	{	lock(lck);
		if (a->a_funct == NULL)
			a->a_funct = f;
		unlock(lck);

		if (a->a_funct != f)
			goto again;
	}

	return(f);
}

Proc	*
findproc(f,m)
Funct	*f;
Atom	*m;
{
	Proc	*p;
	Code	*cp;

	p = f->f_proc;
	if (m == 0)
		m = *a_curr_module;

	while (p != NULL)
	{	if ( p->p_module == m || p->p_flags & P_PUBLIC )
			return(p);
		p = p->p_next;
	}

	/*  make new proc structure  */
	p = (Proc *) alloc_shd_code( procsize + 3 );
	cp = (Code *) p + procsize;
	p->p_code = cp;
	p->p_flags = 0;
	p->p_size = 0;
	p->p_module = m;
	*cp++ = enter_undefined;
	*WordP(cp) = AsFunct(f);
	cp += WordArgSize;

	/*  place new predicate at front of procedure chain  */
	p->p_next = f->f_proc;
	f->f_proc = p;

	return(p);
}

set_proc_flag(Arg, flag)
Word	Arg;
int	flag;
{
	Funct	*f;
	Proc	*p, *op;

	wait_for_argument(Arg);
	if (IsAtom(Arg))
		f = findfunct(AtomVal(Arg), 0);
	else if (IsStruct(Arg))
		f = FunctVal(*StructVal(Arg));
	else
		return(FAIL);

	p = findproc(f, ANULL);
	p->p_flags |= flag;

	/*  put predicate at front of procedure chain  */
	if (f->f_proc != p)
	{	op = f->f_proc;
		while (op->p_next != p)
			op = op->p_next;

		op->p_next = p->p_next;
		p->p_next = f->f_proc;
		f->f_proc = p;
	}
	return(SUCCESS);
}


/*
 *  CODE_TO_FUNCT  --  given a pointer to bytecode locate the functor name
 *		       (stored as arg to the enter_trace instruction).
 */
Funct	*
code_to_funct(c)
Code	*c;
{
	Code	*pc;
	Word	w;

#ifdef CACHE_CODE
	if (c < prv_code_end)	c += PCodeOffset;
#endif
	while (c >= (Code *)shmem_base)
	{	if (*c == enter_trace)
		{	w = *WordP(c+1);
			if (IsFunct(w))
			{	pc = FunctVal(w)->f_proc->p_code;
				if (pc == (c + TraceOffset) || pc == c)
					return(FunctVal(w));
			}
		}
		c--;
	}

	err(1, "code_to_funct: corrupted code\n");
	return((Funct *) 0);	/* keep lint happy ! */
}


