/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include <signal.h>
#include <setjmp.h>
#include "objs.h"
#include "proc.h"
#include "mem.h"
#include "synch.h"
#include "macros.h"
#include "event.h"
#include "ret.h"
#include "trace.h"

#ifdef HERMES
#include <stdio.h>
#include <sys/filio.h>
#include <sys/types.h>
#include "stream.h"
#include "hermes.h"
extern	FILE		*user_error;
extern	runqpt		RQ;
extern	threadpo	prolog_th;
extern	bool		h_debug;
extern	bool		prolog_console_suspended;
extern	void		(*thread_hook)();
extern	time_t		select_time;	/* moved to ../prolog/select.c */
extern	int		select_width;	/* moved to ../prolog/select.c */
extern	int		h_deadlock;	/* moved to ../prolog/select.c */
extern	fd_set		rfdset, wfdset, efdset; /* moved to ../prolog/select.c */
	int		wait_user;
	int		wait_pipe;
#endif
int	Timeslice = 0;			/*  "timeslice" counter  */
#define	N_TIMESLICE	2000		/*  no. reductions per timeslice  */

jmp_buf	slpbuf;
int	hold_intrpt = 0;		/*  hold interrupts  */

extern	int	Ncalls;
extern	Proc	*proc_deadlock;
extern	Proc	*proc_interrupt;
extern	int	*gc_counter;		/*  gc counter  */
extern	Funct	*code_to_funct();


/*
 *  SEARCH_FOR_WORK  --  search for a process to execute
 *		Algorithm:
 *		1. check for an event or (master only) for pending I/O.
 *		2. if timeslice, take from front of local queue
 *		3. take from (back of) private part of local run queue
 *		4. take from public part of local run queue
 *		5. scan remote processors public run queues for work.
 *		   Skip any processor which is not in execute state.
 *		4. if nothing found, check for deadlock
 */
search_for_work()
{
	register char	*lck;
	register Processor *p;

#ifdef CNTS
	PR->st_cnts[0]++;
#endif

restart:
	while (*event)
		if (!execute_event())
			return(0);

	/*  if master and I/O requests are pending, execute them  */
#ifdef HERMES
	wait_user = 0;
#endif
	if (*io_q_front != PNULL && ismaster(PR))
		execute_io();

	PR->pr_state = PR_SEARCH;
	PS = PNULL;

	/*  timeslice - look for work from front of local queue  */
	if (Ncalls >= Timeslice)
	{	Timeslice += N_TIMESLICE;
		lck = PRtolck(PR);
		lock(lck);
		if (PR->q_front != PR->q_back)
		{	PS = *PR->q_front++;
			if (PR->q_front == PR->q_end)
				PR->q_front = PR->q_bot;
		}
		unlock(lck);

		if (PS)
			return(1);
	}

	/*  look for work from back of local queue  */
	if (PR->q_front != PR->q_back)
	{	lck = PRtolck(PR);
		lock(lck);
		if (PR->q_front != PR->q_back)
		{	if (PR->q_back == PR->q_bot)
				PR->q_back = PR->q_end;
			PS = *--PR->q_back;
		}
		unlock(lck);

		if (PS)
			return(1);
	}


/*	fprintf(user_error, "no runnable parlog processes !\n"); */
	/*
	 *  check for deadlock
	 */
	if (ismaster(PR))
	{	/*
		 *  master  -  enters deadlock state and waits for all
		 * 	others to follow suit; if all do then exit with
		 *	deadlock; otherwise return to search state and
		 *	wait for all others to undeadlock.
		 */
		PR->pr_state = PR_DEADLOCK;

		for (p = Pr0+1; p < PrN; p++)
		{
			/*  on Balance this requires -i option to cc  */
			while (p->pr_state == PR_SEARCH)
				continue;

			if (p->pr_state != PR_DEADLOCK)
			{	/*  undeadlock  */
				PR->pr_state = PR_SEARCH;
				for (p = Pr0+1; p < PrN; p++)
				{
					while (p->pr_state == PR_DEADLOCK)
						continue;
				}
				goto restart;
			}
		}

		/*
		 *  check for I/O requests - these must be from user_input
		 *  so this time set stdin to block on read
		 */
		if (*io_q_front != PNULL)
#ifdef HERMES
		{	FILE	*fp = st_user_input->st_fp;
			int	nbytes;

			PR->pr_state = PR_SEARCH;
			if (wait_user)
			{	(void) ioctl(fileno(fp), FIONREAD, &nbytes);
				if (fp->_cnt == 0 && nbytes == 0)
				{	if (h_deadlock & D_PARLOG)
						wait_for_user();
					else
					{	h_deadlock |= D_PARLOG;
						*event |= E_PROLOG;
					}
				}
			}
			else
			{	if (h_deadlock & D_DEADLOCK)
				{	if (nonempty_fdset())
						wait_for_user();
					else
					{
						if (processes_waiting())
						{	*event |= E_PROLOG;
							goto restart;
						}
						deadlock();
						h_deadlock = 0;
						return(1);
					}
				}
				else
					h_deadlock = D_DEADLOCK;
				*event |= E_PROLOG;
			}
			goto restart;
		}
		if (processes_waiting())
		{	*event |= E_PROLOG;
			goto restart;
		}
#else
		{	PR->pr_state = PR_INPUT;
#ifdef DEBUG
			printf(">");
#endif DEBUG
			execute_io();
			PR->pr_state = PR_SEARCH;
			goto restart;
		}
#endif

		/*
		 *  execution is deadlocked.
		 *  master therefore invokes the goal $deadlock/1.
		 */
		deadlock();
#ifdef HERMES
		h_deadlock = 0;
#endif
		return(1);
	}
	return(0);	/* keep lint happy ! */
}


/*
 *  EXECUTE_EVENT  --  process an event.
 *			do event specific initialisation then synchronise.
 *			then call the appropriate event handler.
 *			interrupts are held before barrier so that event
 *			flag cannot change when processors select the event.
 */
execute_event()
{
	/*
	 *  pre-event initialisation
	 */
	PR->pr_state = PR_EVENT;

	if (*event <= E_GC)
	{	/*
		 *  gc: initialise local IPS pointers and gc counter
		 */
		if (ismaster(PR))
			*gc_counter = NPR * sizeof(Mem);
		PR->gc_top = m_st->m_top;
		PR->gc_end = (Word *) (m_st + 1) - NPR;
	}

	hold_intrpt = 1;

	TRACE_event("enter");

	/*  wait for all to arrive here  */
	barrier(bars[0], NPR);

	/*
	 *  now determine the event to execute
	 */
	if (*event >= E_TERM)
#ifdef HERMES
	{	if (*event >= E_PROLOG)
			*event &= ~E_PROLOG;
		else
			TH = (threadpo)0;
		hold_intrpt = 0;
		RQ = RQ->next;
		return(0);
	}
#else
		return(0);
#endif

	else if (*event >= E_INTR)
	{	interrupt();
		if (ismaster(PR))
			*event &= ~E_INTR;
	}
	else if (*event >= E_SGC)
	{	stack_gc();
		if (ismaster(PR))
			*event &= ~E_SGC;
	}
	else if (*event >= E_GC)
	{	heap_gc();
		if (ismaster(PR))
			*event &= ~E_GC;
	}
	else
	{	statistics();
		if (ismaster(PR))
			*event &= ~E_SYNC;
	}

	/*  wait for all to arrive here  */
	barrier(bars[1], NPR);

	/*  re-enable interrupts  */
	if (ismaster(PR))
	{	if (hold_intrpt > 1)
			*event |= E_INTR;
		hold_intrpt = 0;
	}

	TRACE_event("exit");

	return(1);
}

 
/*
 *  C_SYNCH  --  builtin predicate to explictly set event flag to E_SYNC
 */
c_synch()
{
	*event |= E_SYNC;
	return(SUCCESS);
}


/*
 *  DEADLOCK  --  create a process to execute $deadlock/1
 *		All goals suspended on variable are placed in a list argument
 *		to $deadlock/1, and a process is created for this goal.
 */
deadlock()
{
	register Process *sp;
	register Word	*p, *ht;
	register Mem	*mh;
	Process	*dp, *plink;
	Word	savecell, *hngp;
	int	deadp, d;
	Funct	*f;

	/*  temporary measure.. just print out suspended goals  */
	deadp = 0;
	for (mh = m_hb; mh; mh = mh->m_next)
	{	ht = mh->m_top;

		for (p = mh->m_data; p < ht; p++)
		{	if (!IsUnb(*p) || *p == AsUnb(0))
				continue;

			for (sp = UnbVal(*p); sp; sp = UnbVal(plink))
			{	if (ProcessPtr(sp))
					plink = sp->link;
				else
				{	plink = (Process *) ((Word *)sp)[0];
					hngp = (Word *) ((Word *)sp)[1];
					sp = (Process *) ShortVal(*hngp);
				}

				if (!sp || dead(sp))
					continue;

				d = 0;
				for (dp = sp->parent; dp; dp = dp->parent)
				{	if (dead(dp))
					{	d++;
						break;
					}
				}

				if (d)
				{	deadp++;
					continue;
				}

				f = code_to_funct(sp->cont);

				savecell = sp->args[-1];
				sp->args[-1] = AsFunct(f);
				write_term(st_user_output, AsStruct(sp->args - 1), 1);
				fputc('\n', st_user_output->st_fp);

				sp->args[-1] = savecell;
			}
		}
	}
	if (deadp)
		fprintf(st_user_output->st_fp, "%d processes suspended with dead ancestors\n", deadp);

	alloc_ps(PS);
	PS->parent = PNULL;
	PS->cont = proc_deadlock->p_code;
	PS->args = WNULL;
	PS->nargs = 0;
	PS->root = PNULL;
	PS->refcount = 1;

	return;
}


/*
 *  CATCH_ALARM  --  called on receipt of alarm signal.
 *			if sleeping, this reactivates the search for work,
 *			otherwise the signal is effectively ignored.
 */
SIG_T
catch_alarm()
{
	if (PR->pr_state == PR_SLEEP)
		longjmp(slpbuf,0);
}
#ifdef HERMES

 
/*
 *  C_RESUME  --  builtin predicate to explictly set event flag to E_PROLOG
 */
c_resume(Args)
Word	*Args;
{
	threadpo	th, check;

	deref(A0);
	if (IsShort(A0) || IsLong(A0))
		th = (threadpo)int_val(A0);

	if (th)
	{	check = prolog_th;
		while (check != th && check->next != prolog_th)
			check = check->next;
		if (check != th)
		{	bu_error(A0, "resume/1: invalid thread ID");
			return(FAIL);
		}
		(void) add_to_runq(th, FALSE);		/* yac */
	}

	*event |= E_PROLOG;
	return(SUCCESS);
}


/*
 *  C_PROLOG  --  resume Prolog console
 */
c_prolog()
{
	static int parlog_console_suspended = FALSE;

	if (!parlog_console_suspended) {
		parlog_console_suspended = TRUE;
		fprintf(stderr, "\n[ switching to Prolog console ]\n");
		(void) add_to_runq(prolog_th, FALSE);
		*event |= E_PROLOG;
		VarTbl[Nsuspv++] = ToRef(&suspnd(io));
		return(SUSPEND);
	}

	parlog_console_suspended = FALSE;
	return(SUCCESS);
}


processes_waiting()
{
	return(	proc_waiting_for_events() ||
		wait_pipe ||
		!prolog_console_suspended);
}


/*
print_io_q()
{
	Process *p;
	Funct	*f;
	p = *io_q_front;
	while (p != PNULL) {
		f = code_to_funct(p->cont);
		fprintf(user_error, " (%s/%d)", f->f_name->a_string, f->f_arity);
		p = p->link;
	}
	fprintf(user_error, "\n");
}
*/
#endif
