/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include <setjmp.h>
#include <signal.h>
#include "objs.h"
#include "proc.h"
#include "mem.h"
#include "synch.h"
#include "macros.h"
#include "event.h"
#include "trace.h"

extern	Proc	*proc_interrupt;
extern	jmp_buf	inbuf;
extern	int	blocking;		/*  blocking read on user_input	*/
extern	int	hold_intrpt;		/*  hold interrupt during event	*/
extern	Word	**intrpt_var;		/*  interrupt variable	*/
extern	int	Ncalls;			/*  no. calls counter  */



/*
 *  INTERRUPT  --  process a user interrupt.
 *		   All currently runnable processes are suspended on the
 *		   variable argument of $interrupt/1, which is then called.
 */		   
interrupt()
{
	register Process *next, *prev, *last;
	char	*lck;

	TRACE_msg("entering interrupt routine");

	/*
	 *  interrupt: master creates an unbound variable on its heap -
	 *  all runnable processors are suspended on this variable.
	 */
	if (ismaster(PR))
	{	*m_ht->m_top = AsUnb(0);
		*intrpt_var = (m_ht->m_top)++;
	}

	/*
	 *  link together all processes on the local run queue,
	 *  the master processor also adds any processes on the I/O queue;
	 *  then add the chain into the suspension list for intrpt_var.
	 */  
	if (ismaster(PR) && *io_q_front != PNULL)
	{	prev = next = *io_q_front;
		last = *io_q_back;
		*io_q_front = *io_q_back = PNULL;
	}
	else if (PR->q_front != PR->q_back)
	{	prev = PNULL;
		last = *PR->q_front;
	}
	else
		last = PNULL;

	if (last)
	{	while (PR->q_front != PR->q_back)
		{	next = *PR->q_front++;
			if (PR->q_front == PR->q_end)
				PR->q_front = PR->q_bot;

			next->link = prev;
			prev = next;
		}

		if (!ismaster(PR))
		{	/*  wait until interrupt variable ptr is initialised  */
			while( *intrpt_var == WNULL )
				continue;
		}

		lck = ptrtolck(*intrpt_var);
		lock(lck);
		last->link = UnbVal(**intrpt_var);
		**intrpt_var = AsUnb(next);
		unlock(lck);
	}

#ifdef TRACE
	if (ismaster(PR))
	{	int cnt = 0;
        	for (prev = UnbVal(**intrpt_var); prev; prev=prev->link)
        		cnt++;
        	printf("[%d processes now in suspension list]\n", cnt);
        }
#endif TRACE

	/*  master adds a call to $interrupt/1 to its run queue  */
	if (ismaster(PR))
	{	alloc_ps(next);
		next->parent = PNULL;
		next->cont = proc_interrupt->p_code;
		next->args = m_st->m_top;
		next->nargs = 1;
		next->root = PNULL;
		next->refcount = 1;
		*(m_st->m_top)++ = AsRef(*intrpt_var);

		/*  reset interrupt variable ptr  */
		*intrpt_var = WNULL;

		/*  add to runqueue  */
		enqueue_process(next);

		PR->pr_state = PR_EXECUTE;	/*  helps to find process! */
	}
}


/*
 *  CATCH_INTRPT  --  called on receipt of an interrupt signal.
 *			sets the event flag so that interrupt() is called.
 *			if master is in a blocking read, break out of it
 */
SIG_T
catch_intrpt()
{
	if (ismaster(PR))
	{	if (hold_intrpt)
			hold_intrpt++;
		else
			*event |= E_INTR;

		if (blocking)
			longjmp(inbuf,1);
	}
}


#ifdef sun
/*
 *  CATCH_QUIT  --  called on receipt of a quit signal.
 *			sets the event flag to force processors to terminate
 */
SIG_T
catch_quit()
{
	(void) signal(SIGQUIT, SIG_DFL);	/*  if it happens again - quit  */
	if (ismaster(PR))
		*event |= E_TERM;

	(void) execute_event();
	icp_exit(1);
}
#endif


/*
 *  CATCH_BAD  --  called on receipt of a segmentation error
 *			sets the event flag to force processors to terminate
 */
SIG_T
catch_bad()
{
	int	i;

	(void) signal(SIGSEGV, SIG_DFL);	/*  if it happens again - crash  */
	printf("[ERROR: segmentation fault (processor %d)]\n", PR-Pr0);

	if (PR->pr_state == PR_EXECUTE && PS)
	{	printf(" process: %x\n registers: %d\n", PS, PS->nargs);
		for (i=0; i<PS->nargs; i++)
			printf("\t%08x\n", Reg[i]);

		if (m_ht)
			printf(" heap  pointers:  top=%x, ovflw=%x\n",
				m_ht->m_top, m_ht->m_ovflw);
		if (m_st)
			printf(" stack pointers:  top=%x, ovflw=%x\n",
				m_st->m_top, m_st->m_ovflw);
		printf(" reductions: %d\n", Ncalls);
	}

	*event |= E_TERM;
	(void) execute_event();
	icp_exit(1);
}
