/*  Copyright (C) 1990, Jim Crammond, Imperial College. All rights reserved.  */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "objs.h"
#include "proc.h"
#include "mem.h"
#include "macros.h"
#include "synch.h"
#include "instr.h"
#include "ret.h"

extern	Word	atom_system;
extern	Word	atom_nil;
extern	int	path_updated;


/*
 *  OPERATOR SUPPORT
 */
c_set_op_prec(Args)
Word	*Args;
{
	register Op	*op;
	Atom	*opname;
	int	assoc, leftprec, rightprec, i;

	wait_for_argument(A0);
	wait_for_argument(A1);
	if (!IsAtom(A1))
		return(FAIL);
	opname = AtomVal(A1);

	wait_for_argument(A2);
	if (!IsShort(A2) || (assoc = ShortVal(A2)) < 0 || assoc > 2)
		return(FAIL);

	wait_for_argument(A3);
	leftprec = ShortVal(A3);
	wait_for_argument(A4);
	rightprec = ShortVal(A4);

	/*  see if operator is already defined in this set  */
	for (op = opname->a_ops; op != NULL; op = op->op_next)
	{	if (op->op_set == A0)
		{	op->op_larg[assoc] = leftprec;
			op->op_rarg[assoc] = rightprec;
			return(SUCCESS);
		}
	}

	/*  make new operator  */
	op = (Op *) alloc_code(sizeof(Op));

	op->op_set = A0;
	for (i=0; i < 3; i++)
	{	op->op_larg[i] = -1;
		op->op_rarg[i] = -1;
	}
	op->op_larg[assoc] = leftprec;
	op->op_rarg[assoc] = rightprec;

	/*  add to atom's operator chain  */
	op->op_next = opname->a_ops;
	opname->a_ops = op;

	return(SUCCESS);
}

c_get_op_prec(Args)
Word	*Args;
{
	register Op	*op, *op2;
	register int	assoc;

	/*
	 *  look for operator info in given opset or predefined set "system"
	 */
	wait_for_argument(A0);
	wait_for_argument(A1);
	if (!IsAtom(A1))
		return(FAIL);
	op = AtomVal(A1)->a_ops;

	wait_for_argument(A2);
	if (!IsShort(A2) || (assoc = ShortVal(A2)) < 0 || assoc > 2)
		return(FAIL);

	/* try to find operator in named opset first */
	op2 = op;
	while (op != NULL)
	{	if ( op->op_set == A0
		     && op->op_larg[assoc] + op->op_rarg[assoc] > -2 )
		{	return( unify(A3, AsShort(op->op_larg[assoc])) &&
				unify(A4, AsShort(op->op_rarg[assoc])) );
		}
		op = op->op_next;
	}

	/* now look in the system opset */
	while (op2 != NULL)
	{	if ( op2->op_set == atom_system
		     && op2->op_larg[assoc] + op2->op_rarg[assoc] > -2 )
		{	return( unify(A3, AsShort(op2->op_larg[assoc])) &&
				unify(A4, AsShort(op2->op_rarg[assoc])) );
		}
		op2 = op2->op_next;
	}

	/*  no operator precedence found  */
	return(FAIL);
}


static	Word	*h_top, *varbuf;
static	int	nvars, maxvars;


/*
 *  C_VAROCCURS  --  return list of variables in a term
 */
c_varoccurs(Args)
Word	*Args;
{
	Word	*t1;
	
	t1 = h_top = m_ht->m_top;
	varoccurs(A0);
	*h_top++ = atom_nil;
	m_ht->m_top = h_top;
	return(unify(*t1, A1));
}

varoccurs(t1)
register Word	t1;
{
	register Word	*p1;
	int	tag1, n;

	deref(t1);
	tag1 = u_tag(t1);

	switch (tag1)
	{
	case U_REF:
		*h_top = AsList(h_top+1);
		h_top++;
		*h_top++ = t1;
		break;

	case U_STRUCT:
		p1 = StructVal(t1);
		n = FunctVal(*p1)->f_arity;
		while (n-- > 0)
			varoccurs(ToRef(++p1));
		break;

	case U_LIST:
		p1 = ListVal(t1);
		varoccurs(ToRef(p1));
		varoccurs(ToRef(p1+1));
		break;

	case U_CONST:
		break;
	}
}


/*
 *  C_GROUND  --  predicate that succeeds if its argument contains no variables
 */
c_ground(Args)
Word	*Args;
{
	return( ground(A0) );
}

ground(t1)
register Word	t1;
{
	register Word	*p1;
	int	tag1, n;

again:
	deref(t1);
	tag1 = u_tag(t1);

	switch (tag1)
	{
	case U_REF:
		VarTbl[Nsuspv++] = t1;
		return(SUSPEND);

	case U_STRUCT:
		p1 = StructVal(t1);
		n = FunctVal(*p1)->f_arity;
		while (--n > 0)
		{	if (ground(ToRef(++p1)) == SUSPEND)
				return(SUSPEND);
		}

		t1 = ToRef(++p1);
		goto again;

	case U_LIST:
		p1 = ListVal(t1);
		if (ground(ToRef(p1)) == SUSPEND)
			return(SUSPEND);

		t1 = ToRef(++p1);
		goto again;

	case U_CONST:
		return(SUCCESS);
	}

	return(FAIL);	/* keep lint happy ! */
}


/*
 *  C_COPY_TERM  --  builtin predicate to call copy_term()
 */
c_copy_term(Args)
Word	*Args;
{
	Word	t1;
	
	h_top = m_ht->m_top;
	varbuf = m_st->m_top;
	maxvars = (Word *)(m_st + 1) - m_st->m_top;
	nvars = 0;

	t1 = ToRef(h_top++);

	if (copy_term(A0,ToPtr(t1)) >= 0)
	{	m_ht->m_top = h_top;
		return(unify(t1, A1));
	}

	return(FAIL);
}

/*
 *  COPY_TERM  --  make a copy of term t1 which has distinct variables
 *			this version optimises by sharing ground terms.
 */
copy_term(t1,dest)
register Word	t1;
Word	*dest;
{
	register Word	*p1,*p2;
	int	tag1, i, n;
	int	vcnt = 0;

	deref(t1);
	tag1 = u_tag(t1);

	switch (tag1)
	{
	case U_REF:
		/*
		 *  determine whether this is first occurrence of variable.
		 *  previous vars are stored in space above top-of-stack
		 */
		for (i=0; i<nvars; i += 2)
		{	if (varbuf[i] == t1)
			{	*dest = varbuf[i+1];
				break;
			}
		}

		if (i == nvars)
		{	varbuf[nvars++] = t1;
			varbuf[nvars++] = ToRef(dest);
			if (nvars >= maxvars)
				err(1, "copy_term/2: too many variables\n");

			*dest = AsUnb(0);
		}

		vcnt++;
		break;

	case U_LIST:
		*dest = AsList(h_top);
		p1 = ListVal(t1);
		p2 = h_top;
		h_top += 2;

		vcnt += copy_term(ToRef(p1),p2++);
		vcnt += copy_term(ToRef(++p1),p2);

		if (vcnt == 0)
		{	/*  ground term, point to old one & reclaim space  */
			h_top = ListVal(*dest);
			*dest = t1;
			return(0);
		}
		break;

	case U_STRUCT:
		*dest = AsStruct(h_top);
		p1 = StructVal(t1);
		n = FunctVal(*p1)->f_arity;

		p2 = h_top;
		h_top += n + 1;

		*p2++ = *p1++;			/*  functor name  */
		while (--n >= 0)		/*  arguments  */
			vcnt += copy_term(ToRef(p1++),p2++);

		if (vcnt == 0)
		{	/*  ground term, point to old one & reclaim space  */
			h_top = StructVal(*dest);
			*dest = t1;
			return(0);
		}
		break;

	case U_CONST:
		*dest = t1;
		return(0);
	}

	if (h_top > m_ht->m_ovflw)
	{	m_ht->m_top = h_top;
		m_ht = alloc_heap_segment();
		h_top = m_ht->m_data;
	}
	return(vcnt);
}



/*
 *  "FAST MERGE" SUPPORT
 */
extern	Word	atom_nil;


c_create_channel(Args)
Word	*Args;
{
	Process	*ps;

	/*
	 *  create_channel(Id^, Var?)  -
	 *  creates a "channel" to Var (unbound) with id returned in Id.
	 */
	deref(A1);
	if (!IsRef(A1))
	{	bu_error(A1, "create_channel/2: arg must be a variable");
		return(FAIL);
	}

	/*
	 *  allocate a process with an argument that points to the channel
	 *  (this is so that garbage collection works properly).
	 *  normally the process is the ID of the channel
	 */

	alloc_ps(ps);
	ps->args = m_st->m_top++;
	m_st->m_top++;		/* extra cell always points to head of list */
	ps->nargs = 2;
	ps->cont = NULL;

	*(ps->args) = AsRef(A1);
	*(ps->args+1) = AsRef(A1);

	deref(A0);
	if (IsAtom(A0))
	{	if (AtomVal(A0)->a_channel == PNULL)
			AtomVal(A0)->a_channel = ps;
		else
		{	bu_error(A0, "create_channel/2: channel already open");
			return(FAIL);
		}
		return(SUCCESS);
	}

	return( unify(A0, AsShort(ps)) );
}


c_write_channel(Args)
Word	*Args;
{
	Process	*ps, *ps_list;
	Word	*car, *cdr, *ps_word;
	int	ret;
	char	*lck;

	/*
	 *  write_channel(Id?, Item?).
	 *  write an item on the end of the channel given by Id
	 *  Id then points to the new end of channel
	 */
	wait_for_argument(A0);
	if (IsShort(A0))
		ps = (Process *) ShortVal(A0);
	else if (IsAtom(A0) && AtomVal(A0)->a_channel)
		ps = AtomVal(A0)->a_channel;
	else
		return(FAIL);

	/*
	 *  create cons cell on heap with head = Item and tail = unbound,
	 *  then bind tail of channel to this.
	 */
	car = m_ht->m_top++;
	cdr = m_ht->m_top++;
	*car = A1;
	*cdr = AsUnb(0);

	/*  atomically assign channel tail to list  */
	lck = ptrtolck(ps);
	lock(lck);
	ps_word = ToPtr(*ps->args);
	if (IsUnb(*ps_word))
	{	if (ps_list = UnbVal(*ps_word))
			wake(ps_list);
		*ps_word = AsList(car);
		*ps->args = AsRef(cdr);
		ret = SUCCESS;
	}
	else
		ret = FAIL;		/*  channel has been closed  */
	unlock(lck);

	return(ret);
}


c_get_channel(Args)
Word	*Args;
{
	Process	*ps;

	/*
	 *  get_channel(Id?, Var^)  -
	 *  returns the whole history of the "channel" in Var.
	 */
	wait_for_argument(A0);
	if (IsShort(A0))
		ps = (Process *) ShortVal(A0);
	else if (IsAtom(A0) && AtomVal(A0)->a_channel)
		ps = AtomVal(A0)->a_channel;
	else
		return(FAIL);

	return( unify(A1, *(ps->args+1)) );
}


c_close_channel(Args)
Word	*Args;
{
	Process	*ps, *ps_list;
	Word	*ps_word;
	int	ret;
	char	*lck;

	/*
	 *  close_channel(Id?)  -
	 *  close the channel by writing a [] to it.
	 *  Nb cannot reclaim the channel process (yet).
	 */
	wait_for_argument(A0);
	if (IsShort(A0))
		ps = (Process *) ShortVal(A0);
	else if (IsAtom(A0) && AtomVal(A0)->a_channel)
	{	ps = AtomVal(A0)->a_channel;
		AtomVal(A0)->a_channel = PNULL;
	}
	else
		return(FAIL);

	lck = ptrtolck(ps);
	lock(lck);
	ps_word = ToPtr(*ps->args);
	if (IsUnb(*ps_word))
	{	if (ps_list = UnbVal(*ps_word))
			wake(ps_list);
		*ps_word = atom_nil;
		ret = SUCCESS;
	}
	else
		ret = FAIL;		/*  channel has already been closed  */
	unlock(lck);

	return(ret);
}


/*
 *  C_GETENV  --  builtin predicate to read environment variables
 */
c_getenv(Args)
Word	*Args;
{
	char	*val;
	extern	char	*getenv();

	wait_for_argument(A0);
	if (IsAtom(A0)) {
		if ((val = getenv(string_val(A0))))
			return( unify(A1, make_atom(val)) );
		else return(FAIL);
	}
	else {
		bu_error(A0, "getenv/2: arg should be an atom");
		return(FAIL);
	}
}


/*
 *  C_SETENV  --  builtin predicate to set environment variables
 */
c_setenv(Args)
Word	*Args;
{
	char	*sbuf, *s, *val;
	double	f;

	wait_for_argument(A0);
	wait_for_argument(A1);

	sbuf = (char *)malloc(A_STRSIZ);
	if (IsAtom(A0)) {
		strcpy(sbuf, string_val(A0));
		strcat(sbuf, "=");
		s = sbuf + strlen(sbuf);
		if (IsAtom(A1))
			strcpy(s, string_val(A1));
		else if (IsNumber(A1))
		{	if (IsShort(A1))
				(void) sprintf(s, "%d", ShortVal(A1));
			else if (IsLong(A1))
				(void) sprintf(s, "%d", LongVal(A1));
			else
			{	f = FloatVal(A1);
				(void) sprintf(s, "%g", f);
				/*  ensure string contains a decimal point  */
				if (!index(s, '.'))
				{
					if (index(s,'e'))
						(void) sprintf(s, "%.1e", f);
					else	(void) sprintf(s, "%.1f", f);
				}
			}
		}
		else {
			free(sbuf);
			bu_error(A1, "setenv/2: arg should be an atom or number");
			return(FAIL);
		}
#ifdef sun
		if (putenv(sbuf) == 0)
#else
		if (setenv(string_val(A0), s, 1) == 0)
#endif

		{	if (strcmp(string_val(A0), "ICP_PATH") == 0)
#ifdef HERMES
				update_directories();
#else
				path_updated = 1;
#endif
			return(SUCCESS);
		}
		else
		{	free(sbuf);
			return(FAIL);
		}
	}
	else
	{	free(sbuf);
		bu_error(A0, "setenv/2: arg should be an atom");
		return(FAIL);
	}
}

/*
 *  C_CONCAT_ATOM  --  builtin predicate to concatenate a list
 *			of atoms into a single atom.
 */
c_concat_atom(Args)
Word	*Args;
{
	char	sbuf[A_STRSIZ];
	register Word	car, cdr;
	register char	*s;
	double	f;

	wait_for_argument(A0);

	s = sbuf;
	cdr = A0;
	while (cdr != atom_nil)
	{	if (!IsList(cdr))
			return(FAIL);
		car = *ListVal(cdr);
		deref(car);

		if (IsAtom(car))
			strcpy(s, string_val(car));
		else if (IsNumber(car))
		{	if (IsShort(car))
				(void) sprintf(s, "%d", ShortVal(car));
			else if (IsLong(car))
				(void) sprintf(s, "%d", LongVal(car));
			else
			{	f = FloatVal(car);
				(void) sprintf(s, "%g", f);
				/*  ensure string contains a decimal point  */
				if (!index(s, '.'))
				{
					if (index(s,'e'))
						(void) sprintf(s, "%.1e", f);
					else	(void) sprintf(s, "%.1f", f);
				}
			}
		}
		else {
			bu_error(A0, "concat_atom/2: arg should be list of atoms");
			return(FAIL);
		}

		/* find end of string */
		while (*s != '\0')
			s++;

		if ((s - sbuf) >= A_STRSIZ) {
			bu_error(A0, "concat_atom/2: atom exceeds maximum length");
			return(FAIL);
		}

		cdr = *(ListVal(cdr) + 1);
		deref(cdr);
	}

	return( unify(A1, make_atom(sbuf)) );			/*  atom  */
}


/*
 *  C_STAT  --  builtin predicate to get information on a file
 */
c_stat(Args)
Word	*Args;
{
	struct stat	buf;
	char name[A_STRSIZ];

	wait_for_argument(A0);

	if (!IsAtom(A0) || !ic_file_name_stats(string_val(A0), name, 1, &buf))
		return(FAIL);

	return( unify(A1, make_atom(name)) &&
		unify(A2, make_int(buf.st_mtime)) &&
		unify(A3, make_int(buf.st_size))  &&
		unify(A4, make_int(buf.st_mode))  &&
		unify(A5, make_int(buf.st_uid))   &&
		unify(A6, make_int(buf.st_gid)) ) ;
}

/*
 *  C_DEFINED  --  builtin predicate to test if a predicate is defined
 */
c_defined(Args)
Word	*Args;
{
	register Code	*C;

	wait_for_argument(A0);

	if (IsAtom(A0))
		C = findproc(findfunct(AtomVal(A0), 0), ANULL)->p_code;
	else if (IsStruct(A0))
		C = findproc(FunctVal(*StructVal(A0)), ANULL)->p_code;
	else {
		bu_error(A0, "defined/1: argument is not a predicate");
		return(FAIL);
	}

	return(*C != enter_undefined);
}
