/* Copyright (C) 1992 Imperial College */
/*
 * This is the garbage collection routine based on the article
 * 'Garbage Collection for Prolog Based on WAM', which appeared
 * in CACM Vol.31 #6 (June 88)
 *
 * 14/5/92	dac
 * Because of 27-bit integers, the envset is only valid up to the 26th
 * local variable.  All other local variables are ASSUMED to contain
 * valid data.
 *
 * 19/12/91	dac
 * modifications to allow constant and floating-point cells to point
 * at the data instead of the guard cell.
 *
 * 27/3/90	dac
 * fixed bug in sweep_choicepoints.  H should not point at garbage,
 * in particular, it should not point INSIDE constant symbols.
 *
 * 26/3/90	dac (decided fix not required 19/12/91)
 * fixed bug in mark_early_resets.  An early reset is a POINTER from
 * new stack or new heap. (normal stack cells don't count)
 *
 * 23/3/90	dac
 * fixed bug in marking tuples.  I assumed that a tuple was
 * marked if all its components were marked.  This is not true.
 * However, a structure IS marked if the arity cell is marked.
 *
 * 22/3/90	dac
 * fixed bug - HB was not updated after g/c.
 *
 * 22/1/90	dac
 * added optimisation for marking lists and tuples.  Structure is
 * marked if all its components are marked.
 */

#include <time.h>
#include "primitives.h"
#include "gc.h"
#ifdef GNUDOS
#include "dos/clock.h"
#endif
extern int	debugLevel;
extern clock_t	clock();


/* 14/5/92 maximum number of bits in a positive integer */
#define INT_LEN			26

#define inOldHeap(p)		((p) < GC_B->H && (p) >= TH->stacks)
#define inNewHeap(p)		((p) >= GC_B->H && (p) < H)
#define inOldStack(p)		((choicepo)(p) < GC_B && (cellpo)(p) >= BLS)
#define inNewStack(p)		((choicepo)(p) >= GC_B)
#define inOldSegment(p)		(inOldHeap(p) || inOldStack(p))
#define inNewTrail(p)		((p) < GC_B->TR)

#define ptrToNewHeap(p)		(isptr(p) && inNewHeap((cellpo)vl(p)))



static	fourBytes	total_marked;
	fourBytes	reg_marker;

fourBytes	bit[32] = { 0x00000001, 0x00000002, 0x00000004, 0x00000008,
			    0x00000010, 0x00000020, 0x00000040, 0x00000080,
			    0x00000100, 0x00000200, 0x00000400, 0x00000800,
			    0x00001000, 0x00002000, 0x00004000, 0x00008000,
			    0x00010000, 0x00020000, 0x00040000, 0x00080000,
			    0x00100000, 0x00200000, 0x00400000, 0x00800000,
			    0x01000000, 0x02000000, 0x04000000, 0x08000000,
			    0x10000000, 0x20000000, 0x40000000, 0x80000000 };

/* dummy cell used for marking
   NOTE: this cannot be an automatic variable declared inside a
   function, because the address would be too high
 */
static	cell	dummy_cell;


/* does the value field of this cell contain a pointer ? */
bool isptr(p)
cellpo p;
{
    switch (pureTag(p)) {
	case int_ref:
	case nil_ref:
	case guard_ref:
	    return(FALSE);
	case var_ref:
	case float_ref:
	case symb_ref:
	case list_ref:
	case tpl_ref:
	    return(TRUE);
    }
    return(FALSE);
}



void
reset_total_marked()
{
    total_marked = 0L;
}

#define Reverse(current, next)	{ \
	register cellpo temp = (cellpo) vl(next); \
	setval(next, current); \
	current = next; \
	next = temp;}
#define Undo(current, next)	{ \
	register cellpo temp = (cellpo) vl(current); \
	setval(current, next); \
	next = current; \
	current = temp;}
#define Advance(current, next)	{ \
	register cellpo temp = (cellpo) vl(current+1); \
	setval(current+1, next); \
	next = (cellpo) vl(current); \
	setval(current, temp);}
/*
#define Swap3(x, y, z)		{register cellpo temp = x; x = y; y = z; z = temp;}

#define Reverse(current, next)	Swap3(vl(next), current, next)
#define Undo(current, next)	Swap3(vl(current), next, current)
#define Advance(current, next)	Swap3(vl(current+1), next, vl(current))
*/

void
mark_variable(start)
    cellpo	start;
{
    register	cellpo	current = start;
    register	cellpo	next = (cellpo) vl(current);

    head(current);
    total_marked--;		/* don't count stack cells */

forward:
    if (marked(current))
	goto backward;
    mark(current);		/* This is where we mark the cell */
    total_marked++;

    switch (pureTag(current)) {
    case int_ref:
    case nil_ref:
	goto backward;

    case var_ref:
	if (inOldHeap(next) || first(next))	/* end of chain ? */
	   goto backward;
	Reverse(current, next);
	goto forward;

    case symb_ref:
	if (inNewHeap(next)) {
	    next--;
	    if (!marked(next)) {
		twoBytes len = intvl(next);
		mark(next);		/* mark the start of symbol */
		mark(next + len);	/* mark end of symbol */
		total_marked += (len + 1);
	    }
	}
	goto backward;

    case list_ref:
	if (inOldHeap(next) || first(next+1)	/* list pair already marked ? */
	    || (marked(next) && marked(next+1)))	/* dac 22/1/90 */
	   goto backward;
	next++;
	head(next);		/* tail of list is head of a new subchain */
	Reverse(current, next);
	goto forward;

    case tpl_ref:
	{
	register fourBytes size = intvl(next);
	if (inOldHeap(next) || first(next+1)	/* Is structure marked ? */
	    || marked(next))			/* dac 23/3/90 */
	   goto backward;
	while (size--)			/* all components in structure are */
	    head(++next);		/* marked as heads of subchains    */
	Reverse(current, next);
	goto forward;
	}

    case float_ref:
	if (inNewHeap(next)) {
	    next--;
	    if (!marked(next)) {
		mark(next);			/* mark the start of float */
		mark(next + FLOATSIZE + 1);	/* mark end of float */
		total_marked += (FLOATSIZE + 2);
	    }
	}
	goto backward;

    default:
	longjmp(icp_interrupt, 507);
    }

backward:
    if (!first(current)) {	/* internal cell ? */
	Undo (current, next);
	goto backward;
    }

    /* this is a head of chain */
    unhead(current);
    if (current == start) {	/* back to the beginning ? */
	/* dac 19/12/91 check for adjusted current pointer */
	if (next+1 == (cellpo)vl(current))
	    setval(current, next);
	return;
    }
    current--;			/* look at the next component of structure */
    Advance(current, next);
    goto forward;
}

/*------------------------------------------------------------*/

mark_registers(regs)
    int		regs;
{
    /* NOTE: must not declare dummy_cell here, it must be a global variable ! */

    while (regs--) {		/* reg is number of active A registers */
	/* dac 19/12/91 */
	if (ptrToNewHeap(A+regs))
	    mark_variable(A+regs);
    }
}

mark_environments(env, cp)
    envpo	env;
    codepo	cp;
{
    cellpo	v;
    short	size;
    register	fourBytes	*mask, var_set;

    while (inNewStack(env)) {
	if (cp) {
	    size = envsize(cp);
	    var_set = envset(cp);
	    mask = &bit[size];

	    while (size--)
		if (size >= INT_LEN || (*--mask & var_set)) {	/* dac 14/5/92 */
		    v = &env->Y[size];
		    if (marked(v))
			return;
		    else if (ptrToNewHeap(v))
			mark_variable(v);
		}
	}

	cp = env->CP;
	env = env->CE;
    }
}

mark_trail()
{
    /* NOTE: must not declare dummy_cell here, it must be a global variable ! */

    cellpo	t, p;

    for (t = TR; inNewTrail(t); t++) {
	p = (cellpo) vl(t);
	if (inOldSegment(p)) {
	    dummy_cell = *p;
	    mark_variable(&dummy_cell);
	    /* dac 19/12/91 */
	    unmark(&dummy_cell);
	    *p = dummy_cell;
	}
    }
}

mark_early_resets(chpt, tr, trailcells_deleted)
    choicepo	chpt;
    cellpo	*tr;
    fourBytes	*trailcells_deleted;
{
    register	cellpo	p;

    while (*tr < chpt->TR) {
	p = (cellpo) vl(*tr);
	if (p && !inOldSegment(p) && !marked(p)
			&& ptrToNewHeap(p)) {	/* dac 26/3/90 */
	    mkunb(p);
	    setval(*tr, NULL);		/* so we can recognise it later */
	    (*trailcells_deleted)++;
	}
	(*tr)++;
    }
}

mark_choicepoints(trailcells_deleted)
    fourBytes	*trailcells_deleted;
{
    choicepo	chpt = B;
    cellpo	t = TR, p;
    short	argcnt;

    while (inNewStack(chpt) && chpt > (choicepo)BLS) {
 	mark_early_resets(chpt, &t, trailcells_deleted);
	mark_environments(chpt->E, chpt->CP);
	argcnt = chpt->AX;
	for (p = (cellpo)chpt - argcnt; argcnt--; p++)
	    if (ptrToNewHeap(p))
		mark_variable(p);
	chpt = chpt->B;
    }
}

collect_trail(trailcells_deleted)
    fourBytes	*trailcells_deleted;
{
    register	cellpo	dest = GC_B->TR - 1;
    register	cellpo	current;
    register	cellpo	t = TR;
    register	choicepo chpt = B;

    /* update choicepoints */

    while (inNewStack(chpt)) {
	for (; t < chpt->TR; t++)
	    if (vl(t) == NULL)		/* early reset variable */
		(*trailcells_deleted)--;
	chpt->TR += *trailcells_deleted;
	chpt = chpt->B;
    }

    /* compact trail */

    for (current=GC_B->TR-1; current>=TR; current--)
	if (vl(current) != NULL) {
	    setval(dest, vl(current));
	    dest--;
	}
    TR = dest + 1;
}

/*------------------------------------------------------------*/

into_relocation_chain(j, current)
    cellpo	j, current;
{
    setval(current, vl(j));
    if (first(j))
	head(current);
    else unhead(current);
    setval(j, current);
    head(j);
}

update_relocation_chain(current, dest)
    cellpo	current, dest;
{
    register	cellpo	j;

    while(first(current)) {
	j = (cellpo) vl(current);
	setval(current, vl(j));
	if (first(j))
	    head(current);
	else unhead(current);
	setval(j, dest);
	unhead(j);
    }
}

push_registers(regs)
    int		regs;
{
    reg_marker = 0L;

    while (regs--)		/* reg is number of active A registers */
	if (ptrToNewHeap(&A[regs])) {
	    *--TR = A[regs];	/* put on trail to facilitate chaining */
	    reg_marker |= bit[regs];	/* note which regs have been pushed */
	}
}

sweep_trail()
{
    register	cellpo	current, p;

    for (current=TR; inNewTrail(current); current++) {
	p = (cellpo) vl(current);
	if (inNewHeap(p))
	    into_relocation_chain(p, current);
	else if (inOldSegment(p) && ptrToNewHeap(p))
	    into_relocation_chain((cellpo) vl(p), p);
    }
}

sweep_environments(env, cp)
    envpo	env;
    codepo	cp;
{
    cellpo	v;
    short	size;
    register	fourBytes	*mask, var_set;

    while (inNewStack(env)) {
	if (cp) {
	    size = envsize(cp);
	    var_set = envset(cp);
	    mask = &bit[size];

	    while (size--)
		if (size >= INT_LEN || (*--mask & var_set)) {	/* dac 14/5/92 */
		    v = &env->Y[size];
		    if (ptrToNewHeap(v)) {
			if (!marked(v))	/* we have already been here */
			    return;
			else {
			    unmark(v);
			    into_relocation_chain((cellpo) vl(v), v);
			}
		    }
		}
	}

	cp = env->CP;
	env = env->CE;
    }
}

sweep_choicepoints()
{
    register	cellpo	v;
    register	choicepo chpt = B;
    register	short	argcnt;
    register	cellpo	addrH;
 

    while (inNewStack(chpt) && chpt > (choicepo)BLS) {
	sweep_environments(chpt->E, chpt->CP);
        argcnt = chpt->AX;
        for (v = (cellpo)chpt - argcnt; argcnt--; v++)
	    if (ptrToNewHeap(v)) {
		unmark(v);
		into_relocation_chain((cellpo) vl(v), v);
	    }

	v = chpt->H;

	/* adjust H if pointing at garbage */
	while (!marked(v) && v < H) {
	    if (IsGuard(v)) {
		v += intvl(v);
	    }
	    v++;
	}

	if (v == H) {		/* create a dummy cell on the heap */
	    mknil(v);
	    mark(v);
	    total_marked++;
	    H++;
	}

	addrH = &chpt->gcH;
	chpt->gcH = (cell)v;
	into_relocation_chain(v, addrH);
	chpt = chpt->B;
    }
}

void
compact_segment(start, end)
    cellpo	start, end;
{
    register	cellpo	dest = start + total_marked - 1;
    register	cellpo	current = end;
    /* the upward phase */
    while (--current >= start) {
	if (IsGuard(current)) {		/* untagged data */
	    int untaggedsize = intvl(current);
	    current += untaggedsize;
	    if (marked(current)) {
		dest += untaggedsize;
		update_relocation_chain(current, dest+1);
		dest--;
	    }
	    continue;
	}
	if (marked(current)) {
	    update_relocation_chain(current, dest);
	    if (ptrToNewHeap(current)) {
		if ((cellpo) vl(current) < current)
		    into_relocation_chain((cellpo) vl(current), current);
		else if (current == (cellpo) vl(current))  /* cell pointing to itself */
		    setval(current, dest);
	    }
	    dest--;
	}
    }

    /* the downward phase */
    dest = start;
    for (current=dest; current<end; current++) {
	if (marked(current)) {
	    if (IsGuard(current))
		update_relocation_chain(current, dest+1);
	    else
		update_relocation_chain(current, dest);
	    if (ptrToNewHeap(current) && (cellpo) vl(current) > current) {
		/* move the current cell and insert into chain */
		into_relocation_chain((cellpo) vl(current), dest);
		*dest = tg(current) | first(dest) | vl(dest);
	    }
	    else {
		/* just move the current cell */
		unmark(current);
		*dest = *current;
		if (IsGuard(current)) {
		    twoBytes len = intvl(current);
		    while (len--)	/* copy untagged data */
			*++dest = *++current;
		    unmark(dest);	/* unmark the tail guard cell */
		}
	    }
	    dest++;
	}
	else if (IsGuard(current)) {	/* unmarked untagged data */
		current += intvl(current);
        }
    }

    H = dest;
    if (H != start + total_marked)
	(void) fprintf(stderr, "error in garbage collection\n");
}

compact_heap()
{
    register	choicepo chpt = B;

    compact_segment(GC_B->H, H);

    /* update heap tops */
    while (inNewStack(chpt) && chpt > (choicepo)BLS) {
	register cellpo addr = &chpt->gcH;
	/* if pointing to untagged data, adjust to point to guard head */
	register cellpo hcell = (cellpo)vl(addr) - 1;
	if (IsGuard(hcell) && intvl(hcell) > 0)	/* found a guard head */
	    chpt->H = hcell;
	else
	    chpt->H = (cellpo) vl(addr);
	chpt = chpt->B;
    }
    HB = B->H;		/* dac 22/3/90 */
}
	    
pop_registers(regs)
    int		regs;
{
    register	int	i;
    for (i=0; i<regs; i++)		/* reg is number of active A registers */
	if (reg_marker & bit[i]) { 	/* was this reg pushed ? */
	    A[i] = *TR++;		/* restore values from trail */
	    unmark(A+i);
	}
}

/*------------------------------------------------------------*/

marking_phase(regs)
    int		regs;
{
    fourBytes	trailcells_deleted = 0L;

    reset_total_marked();
    mark_registers(regs);
    mark_environments(E, CP);
    mark_trail();
    mark_choicepoints(&trailcells_deleted);
    collect_trail(&trailcells_deleted);
}

compaction_phase(regs)
    int		regs;
{
    push_registers(regs);
    sweep_trail();
    sweep_environments(E, CP);
    sweep_choicepoints();
    compact_heap();
    pop_registers(regs);
}

/*------------------------------------------------------------*/

fourBytes
collect_garbage(regs)
    int		regs;
{
    fourBytes	cells;
    cellpo	initialH = H;
#ifdef DEBUG
    fourBytes	stackCells;
#endif
    double	t1, t2;

    debug(3, printf("\ncalling garbage collector\n"));

    /* fprintf(stderr, "\ncalling garbage collector\n"); */

    t1 = usertime();
    marking_phase(regs);
    t2 = usertime();
    t1 = t2 - t1;

    debug2(4, printf("finished marking ...\n"));
    debug2(4, debugger(CP));

    compaction_phase(regs);

    GC_B = B;		/* must do this here for segmented g/c to work */
    t2 = usertime() - t2;

    cells = initialH - H;

    TH->stats.gc_time += (t1 + t2);
    TH->stats.gc_count++;
    TH->stats.gc_acc += (int)(cells * 100 / (TH->BLS - TH->stacks));
    TH->stats.starttime += (t1 + t2);
    TH->stats.lasttime += (t1 + t2);

    debug(3, stackCells = normal(TR) - normal(max(E+1, (envpo)(B+1))));
    debug(3, printf("Free Stack Space : %ld (%2d%%)\n", stackCells,
		(int)(stackCells * 100 / (TH->TSZ - (TH->BLS - TH->stacks)))));
    debug(3, printf("#Cells Reclaimed : %ld (%2d%%)\n",
		cells, (int)(cells * 100 / (TH->BLS - TH->stacks))));
    debug(3, printf("Total Time Taken : %.4f sec\n", (t1 + t2)));

    return(cells);
}

/*  used for debugging only
confirm(n)
int n;
{
    register	choicepo chpt = B;
    register	cellpo	addrH;
    cellpo	v;

    (void) fprintf(stdout, "---BEGIN %d ---\n", n);
    while (inNewStack(chpt) && chpt > (choicepo)BLS) {
	v = &chpt->gcH;
	(void) fprintf(stdout, "[%8lx]", v);
	while (first(v)) {
		(void) fprintf(stdout, " -> %8lx", *v);
		v = vl(v);
	}
	(void) fprintf(stdout, " -> %8lx", *v);
	(void) fprintf(stdout, " = ");
	prtcell(v, TH);
	(void) fprintf(stdout, "\n");
	chpt = chpt->B;
    }
    (void) fprintf(stdout, "---  END %d ---\n\n", n);
}
*/
