/* Copyright (C) 1992 Imperial College */
/*
 *  Main emulator program. Emulates the WAM style machine.
 *
 */

#ifdef PROFILE
#include <prof.h>
#else
#define MARK(X)
#endif

#ifndef GNUDOS /* No standard ioctl for DOS */
#include <sys/ioctl.h>
#endif
#include "icp.h"
#include "primdef.h"
#include "arith.h"
#include "io.h"
#include "opcodes.h"
#include "symtab.h"
#include "dynamic.h"
#include "foreign.h"
#include "events.h"



/* external references */

extern bool	resume();
extern codepo	search_symbol();	/* look up address of a given relation */
extern bool	samesymb();		/* compare two constants by their print name */
extern void	guessLocals();		/* set number and current usage of local vars */
extern void	printName();		/* for displaying symbols */
extern opcode	get_brkcode();		/* original instruction */
extern fourBytes collect_garbage();	/* garbage collector */
extern void	interrupt_handler();	/* interrupt handlers */

extern codepo	boot;			/* base of the code space */
extern int	debugLevel, oldLevel;	/* for setting granularity of trace */
extern short	numRegs, numLocals;	/* number of regs and local vars displayed in trace */
extern opcode	brkcode[];		/* global variable which contains the broken code */
extern fourBytes trace[];		/* a table of the most recently executed instructions */
extern short	tracePtr;		/* next free slot in trace table */
extern symbpo	interrupt_sym;		/* a ptr to the symbol '$interrupt$' */
extern symbpo	undef_sym;		/* a ptr to the symbol '$undefined$' */
extern eventpt	eventq;			/* event queue */
extern runqpt	RQ;			/* run queue */
extern threadpo	prolog_th;		/* thread ID of top level */
extern int	h_deadlock;
#ifdef HERMES
extern threadpo	parlog_th;		/* thread ID of parlog thread */
extern	bool	h_debug;
#else
	bool	h_debug = FALSE;
#endif

static struct {				/* push-down list for unification */
    cellpo	S;
    rwmode	mode;
} pdl[PDLSIZE], *pdltop = pdl, *pdlmax = pdl + PDLSIZE;

cell		A[REGISTERS];		/* argument registers */


threadpo	TH;			/* the current thread */
codepo		sP,			/* program counter (shadow) */
		CP,			/* continuation program counter */
		ERR_PT=0;		/* error label */
cellpo		H,
		HB,
		HMAX,
		CHMAX,
		BLS,
		TR,			/* trail point */
		S;			/* structure pointer */
choicepo	B,			/* last choice point */
		SB,			/* where to cut */
		CSB,			/* continuation cut */			
		GC_B,			/* last garbage collected choice point */
		CATCH;
envpo		E,			/* local environment */
		CE;			/* continuation local environment */
twoBytes	ERROR = 0;		/* error code */
ufourBytes 	C, CC;			/* clock and clock register */

void		(*thread_hook)() = NULL; /* function to call before running thread */

iopo		current_input,		/* I/O descriptor for current input */
		current_output;		/* I/O descriptor for current output */

jmp_buf		icp_interrupt;

#define mkcodeflt(Addr, Ptr)	mkflt(Addr, Ptr)

#define DEFput_x_var(I, J) \
	{ \
	mkunb(H); \
	A[I] = A[J] = *H++; \
	continue;}

#define DEFput_y_var1(I, J) \
	{register cellpo ptr = &(E->Y[I]); \
	mkunb(ptr); \
	A[J] = *ptr; \
	continue;}
#define DEFput_y_var2(J) \
	{register cellpo ptr = &(E->Y[*P++]); \
	mkunb(ptr); \
	A[J] = *ptr; \
	continue;}

#define DEFput_y_val1(I, J) \
	{ \
	A[J] = E->Y[I]; \
	continue;}
#define DEFput_y_val2(J) \
	{ \
	A[J] = E->Y[*P++]; \
	continue;}

#define DEFput_unsafe_y1(I, J) \
	{register cellpo ptr = &(E->Y[I]); \
	delnk(ptr); \
	if (IsVar(ptr) && ptr > (cellpo)E) {	/* unsafe variable */ \
		mkunb(H); \
		mkreset(ptr); \
		*ptr = *H++; \
	} \
	A[J] = *ptr; \
	continue;}
#define DEFput_unsafe_y2(J) \
	{register cellpo ptr = &(E->Y[*P++]); \
	delnk(ptr); \
	if (IsVar(ptr) && ptr > (cellpo)E) {	/* unsafe variable */ \
		mkunb(H); \
		mkreset(ptr); \
		*ptr = *H++; \
	} \
	A[J] = *ptr; \
	continue;}

#define DEFput_void(I) \
	{ \
	mkunb(H); \
	A[I] = *H++; \
	continue;}

#define DEFput_const(I) \
	{ \
	mksymb(&A[I], (fourBytes)rel(P)); \
	P++; \
	continue;}

#define DEFput_int(I) \
	{ \
	mkint1(&A[I], *(fourBytes *)P); \
	P += 2; \
	continue;}

#define DEFput_nil(I) \
	{ \
	mknil(&A[I]); \
	continue;}

#define DEFput_list(I) \
	{ \
	alloc_list(&A[I], S); \
	continue;}

#define DEFput_tpl(I) \
	{ \
	alloc_tpl(&A[I], S, *P); \
	P++; \
	continue;}

#define DEFput_slash(I) \
	{ \
	mkint(&A[I], SB); \
	continue;}

#define DEFget_x_var(I, J) \
	{ \
	A[J] = A[I]; \
	continue;}

#define DEFget_y_var1(I, J) \
	{ \
	E->Y[J] = A[I]; \
	continue;}
#define DEFget_y_var2(I) \
	{ \
	E->Y[*P++]=A[I]; \
	continue;}

#define DEFget_x_val(I, J) \
	{register cellpo ptr; \
	T1 = &A[I]; \
	T2 = ptr = &A[J]; \
	delnk(ptr); \
	*T2 = *ptr;			/* dereferenced result left in reg */ \
	goto unify_loop;}

#define DEFget_y_val1(I, J) \
	{ \
	T1 = &A[I]; \
	T2 = &(E->Y[J]); \
	goto unify_loop;}
#define DEFget_y_val2(I) \
	{ \
	T1 = &A[I]; \
	T2 = &(E->Y[*P++]); \
	goto unify_loop;}

#define DEFget_const(I) \
	{register cellpo ptr = &A[I]; \
	register fourBytes temp = (fourBytes)rel(P); \
	delnk(ptr); \
	if (IsVar(ptr)) { \
		mkreset(ptr); \
		mksymb(ptr, temp); \
	} else if (NotSymb(ptr) || !samesymb(symbvl(ptr), (symbpo)temp)) \
		goto fail_label; \
	P++; \
	continue;}

#define DEFget_int(I) \
	{register cellpo ptr = &A[I]; \
	delnk(ptr); \
	if (IsVar(ptr)) { \
		mkreset(ptr); \
		mkint1(ptr, *(fourBytes *)P); \
	} else if (NotInt(ptr) || intvl(ptr) != *(fourBytes *)P) \
		goto fail_label; \
	P += 2; \
	continue;}

#define DEFget_nil(I) \
	{register cellpo ptr = &A[I]; \
	delnk(ptr); \
	if (IsVar(ptr)) { \
		mkreset(ptr); \
		mknil(ptr); \
	} else if (NotNil(ptr)) \
		goto fail_label; \
	continue;}

#define DEFget_list(I) \
	{register cellpo ptr = &A[I]; \
	delnk(ptr); \
	if (IsVar(ptr)) { \
		mkreset(ptr); \
		alloc_list(ptr, S); \
		mode = write_mode; \
	} else if (IsList(ptr)) { \
		S = (cellpo) vl(ptr);	/* go into the list pair */ \
		mode = read_mode; \
	} else \
		goto fail_label; \
	continue;}

#define DEFget_tpl(I) \
	{register cellpo ptr = &A[I]; \
	register int temp = *P++; \
	delnk(ptr); \
	if (IsVar(ptr)) { \
		mkreset(ptr); \
		alloc_tpl(ptr, S, temp); \
		mode = write_mode; \
	} else if (IsTpl(ptr)) { \
		S = (cellpo) vl(ptr);	/* go into the tuple */ \
		if (intvl(S) != temp) \
			goto fail_label; \
		S++;		/* skip over the size cell of the tuple */ \
		mode = read_mode; \
	} else \
	goto fail_label; \
	continue;}

#define DEFunify_x_var(I) \
	{ \
	if (mode == write_mode) \
		mkunb(S); \
	A[I] = *S++; \
	continue;}

#define DEFunify_y_var(I) \
	{ \
	if (mode == write_mode) \
		mkunb(S); \
	E->Y[I] = *S++; \
	continue;}

#define DEFunify_x_val(I) \
	{ \
	if (mode == read_mode) { \
		T1 = T2 = &A[I]; \
		delnk(T1); \
		*T2 = *T1;		/* dereferenced result left in reg */ \
		T1 = S++; \
		goto unify_loop; \
	} else \
		*S++ = A[I]; \
	continue;}

#define DEFunify_y_val(I) \
	{ \
	if (mode == read_mode) { \
		T1 = S++; \
		T2 = &E->Y[I]; \
		goto unify_loop; \
	} else \
		*S++ = E->Y[I]; \
	continue;}

#define DEFunify_loc_x_val(I) \
	{ \
	if (mode == read_mode) { \
		T1 = T2 = &A[I]; \
		delnk(T1); \
		*T2 = *T1;		/* dereferenced result left in reg */ \
		T1 = S++; \
		goto unify_loop; \
	} else { \
		register cellpo ptr = &A[I]; \
		delnk(ptr); \
		if (IsVar(ptr) && ptr > BLS) { \
			mkunb(S); \
			mkreset(ptr); \
			*ptr = *S++; \
		} else \
			*S++ = *ptr; \
	} \
	continue;}

#define DEFunify_loc_y_val(I) \
	{ \
	if (mode == read_mode) { \
		T1 = S++; \
		T2 = &E->Y[I]; \
		goto unify_loop; \
	} else { \
		register cellpo ptr = &E->Y[I]; \
		delnk(ptr); \
		if (IsVar(ptr) && ptr > BLS) { \
			mkunb(S); \
			mkreset(ptr); \
			*ptr = *S++; \
		} else \
			*S++ = *ptr; \
	} \
	continue;}

#define DEFset_x_var(I) \
	{ \
	mkunb(S); \
	A[I] = *S++; \
	continue;}

#define DEFset_y_var(I) \
	{ \
	mkunb(S); \
	E->Y[I] = *S++; \
	continue;}

#define DEFset_x_val(I) \
	{ \
	*S++ = A[I]; \
	continue;}

#define DEFset_y_val(I) \
	{ \
	*S++ = E->Y[I]; \
	continue;}

#define DEFset_loc_x_val(I) \
	{register cellpo ptr = &A[I]; \
	delnk(ptr); \
	if (IsVar(ptr) && ptr > BLS) { \
		mkunb(S); \
		mkreset(ptr); \
		*ptr = *S++; \
	} else \
		*S++ = *ptr; \
	continue;}

#define DEFset_loc_y_val(I) \
	{register cellpo ptr = &E->Y[I]; \
	delnk(ptr); \
	if (IsVar(ptr) && ptr > BLS) { \
		mkunb(S); \
		mkreset(ptr); \
		*ptr = *S++; \
	} else \
		*S++ = *ptr; \
	continue;}

#define DEFput_float(I) \
	{ \
	mkcodeflt(&A[I], P); \
	P += 4; \
	continue;}

#define DEFget_float(I) \
	{register cellpo ptr = &A[I]; \
	delnk(ptr); \
	if (IsVar(ptr)) { \
		mkreset(ptr); \
		mkcodeflt(ptr, P); \
	} else if (NotFloat(ptr) || !numequal(floatvl(ptr), *(FLOAT *)P)) \
		goto fail_label; \
	P += 4; \
	continue;}


void
goal_to_term(pred, num)
symbpo pred;
twoBytes num;
{
    register
    cellpo	ptr,
		argPtr,
		regs = A;

    /* construct the goal in A[0] */
    alloc_tpl(A, argPtr, num+1);
    mksymb(argPtr, pred);
    while (num--) {
	ptr = ++regs;
	delnk(ptr);
	if (IsVar(ptr) && (cellpo) vl(ptr) >= BLS) {
	    argPtr++;
	    mkunb(argPtr);
	    mkreset(ptr);
	    *ptr = *argPtr;
	}
	else *++argPtr = *ptr;
    }
}



/*
 * main emulator loop
 */
solve(P)
register codepo	P;
{
    register
    rwmode	mode;		/* read or write mode */

    cellpo	T1, T2;		/* special variables used for the unify loop */

    bool	(*c_funct)();	/* foreign function */
    codepo	PP;		/* shadow program counter */

    if (thread_hook)
	(*thread_hook)();

  for (;;) {
    int err;
    if (err = setjmp(icp_interrupt)) {	/* aborts are handled here */
	ERROR = err;
	B = CATCH;
	debug(1, printf("\nInterrupt ...\n"));
	goto fail_label;
    }
    else
    for (;;) {
#ifdef DEBUG
	opcode Instr;
#endif
	MARK(FETCH);
	debug(2, debugger(P));

/* 22/6/90 dac	Commented out for performance.  Hardly ever used anyway !!
	debug(0, trace[tracePtr++] = PC(P);
		 if (tracePtr == AUDITSIZE) tracePtr = 0);
*/

#ifdef DEBUG
/* 	(void)printf("%ld: ", P); oneInstruction(P); */
	Instr = *P++;
next_instr:
	switch (Instr)
#else
	switch (*P++)
#endif
	{

	case halt:			/* stop execution */
	    MARK(halt);
	    P--;
	    debugLevel = 2;
	    debugger(P);
	    continue;

	case succ:			/* succeed clause */
	    MARK(succ);
	    HMAX = CHMAX;
	    SB = CSB;
	    P = CP;			/* continue from parent call */
	    debug(0, numRegs = DISPLREG; guessLocals(P));
	    debug2(1, printf("\nsucc --- P=%ld\n", PC(P)));
	    continue;

	case jmp:			/* LBL local last call */
	    MARK(jmp);
	    HMAX = CHMAX;
	    SB = B;
	    P += *P;			/* adjust the program counter */
	    debug2(1, printf("\njmp --- P=%ld\n", PC(P));
		      dmpA(A, TH);
		      debugger(P));
	    continue;

	case execute:			/* PROC external last call */
	    MARK(execute);
	    HMAX = CHMAX;
	    SB = B;
	    PP = P;
	    if(!(P = ext_dict(rel(P))->addr)) {
		/* IMPORTANT !  Do not alter the value of PP */
		if (c_funct = search_foreign(ext_pred(rel(PP)), ext_arity(rel(PP))))
		    goto foreign_code;
		else {
		    goal_to_term(ext_pred(rel(PP)), ext_arity(rel(PP)));
		    P = search_symbol(undef_sym, 0);
		}
            }
	    else {
		debug2(1, PP = rel(PP);
			  (void)printf("\nexecute ");
			  printName((symbpo)ext_pred(PP), stdout);
			  (void)printf("/%d --- P=%ld\n", ext_arity(PP), PC(P));
			  dmpA(A, TH);
			  debugger(P));
	    }
	    continue;

	case executev:			/* ARITY meta last call */
	    MARK(executev);
	{
	    register cellpo ptr;

	    debug(0, numRegs = *P+1; numLocals = 0);
	    SB = B;
	    ptr = A;
	    delnk(ptr);
	    if (NotSymb(ptr))
		goto fail_label;
	    /* if relation is defined, adjust the program counter */
	    if (!(P=search_symbol(symbvl(ptr),*(PP=P)))) {
		if (IsSymb(ptr) && (c_funct = search_foreign(symbvl(ptr), *PP)))
		    goto foreign_code;
		else {
		    goal_to_term(symbvl(ptr), *PP);
		    P = search_symbol(undef_sym, 0);
		}
	    }
	    debug2(1, printf("\nexecutev --- P=%ld\n", PC(P));
		      dmpA(A, TH);
		      debugger(P));
	    continue;
	}

	case jsr:			/* LBL,LOCAL,BITMAP call local procedure */
	    MARK(jsr);
	    CSB = SB;
	    SB = B;
	    CP = P + 4;			/* where to return to */
	    P += *P;			/* enter into the procedure */
	    debug(0, numRegs = DISPLREG; numLocals = 0);
	    debug2(1, printf("\njsr --- P=%ld\n", PC(P));
		      dmpA(A, TH);
		      debugger(P));
	    continue;

	case callme:			/* LBL,LOCAL,BITMAP inline call */
	    MARK(callme);
	    CSB = SB;
	    SB = B;
	    CP = rel(P);		/* where to return to */
	    P += 4;			/* enter into the procedure */
	    debug(0, numRegs = DISPLREG; numLocals = 0);
	    debug2(1, printf("\ncallme --- P=%ld\n", PC(P));
		      dmpA(A, TH);
		      debugger(P));
	    continue;

	case call:			/* PROC,LOCALS,BITMAP call an external procedure */
	    MARK(call);

	    /* check for events */
	    if (eventq) {
		switch (event_handler(TRUE)) {
		    case RETURN:
			P--;
			sP = P;
#ifdef HERMES
			if (TH && TH != parlog_th)
#else
			if (TH)
#endif
				save_thread(TH);
			RQ = RQ->next;
			h_deadlock = 0;
			return(-1);
			break;
		    case INTERRUPT:
			CSB = SB;
			SB = B;
			CP = P + 4;
			goal_to_term(ext_pred(rel(P)), ext_arity(rel(P)));
			P = search_symbol(interrupt_sym, 0);
			continue;
			break;
		    case PROCEED:
		    default:
			break;
		}
	    }
	    CP = P + 4;

	    PP = P;
	    if (!(P = ext_dict(rel(P))->addr)) {
		/* IMPORTANT !  Do not alter the value of PP */
		if (c_funct = search_foreign(ext_pred(rel(PP)), ext_arity(rel(PP))))
		    goto foreign_code;
		else {
		    goal_to_term(ext_pred(rel(PP)), ext_arity(rel(PP)));
		    P = search_symbol(undef_sym, 0);
		}
	    }

	    CSB = SB;
	    SB = B;

	    debug(0, numRegs = DISPLREG; numLocals = 0);
	    debug2(1, PP = rel(PP);
		      (void)printf("\ncall ");
		      printName((symbpo)ext_pred(PP), stdout);
		      (void)printf("/%d --- P=%ld\n", ext_arity(PP), PC(P));
		      dmpA(A, TH);
		      debugger(P));
	    continue;

	case callv:			/* ARITY,LOCALS,BITMAP meta call */
	    MARK(callv);
	{
	    register cellpo ptr;
	    debug(0, numRegs = *P+1; numLocals = 0);
	    CSB = SB;
	    SB = B;
	    CP = P + 4;
	    ptr = A;
	    delnk(ptr);
	    if (NotSymb(ptr))
		goto fail_label;
	    /* if relation is defined, adjust the program counter */
	    if (!(P=search_symbol(symbvl(ptr),*(PP=P)))) {
		if (IsSymb(ptr) && (c_funct = search_foreign(symbvl(ptr), *PP)))
		    goto foreign_code;
		else {
		    goal_to_term(symbvl(ptr), *PP);
		    P = search_symbol(undef_sym, 0);
		}
	    }
	    debug2(1, printf("\ncallv --- P=%ld\n", PC(P));
		      dmpA(A, TH);
		      debugger(P));
	    continue;
	}

	case allocate:			/* allocate a call record and environment */
	    MARK(allocate);
	{
	    register cellpo ptr;

	    CE = E;			/* 'save' the old environment */

	    debug(3, printf("allocate --- E=0x%0l5x, B=0x%05lx, ", normal(E), normal(B)));
	    ptr = (choicepo)CE > B ? (cellpo)(CE+1)+envsize(CP) : (cellpo)(B+1);
	    if (ptr > TR - FUZZ)
		longjmp(icp_interrupt, 504);

	    ((envpo)ptr)->CP = CP;	/* store next address */
	    ((envpo)ptr)->CSB = CSB;	/* store continuation cut */
	    ((envpo)ptr)->CE = CE;	/* store old continuation environment */
	    ((envpo)ptr)->HMAX = CHMAX;	/* pre-allocated heap space */
	    E = (envpo)ptr;
	    debug(3, printf("E now at 0x%0l5x\n", normal(E)));
	    debug(0, guessLocals(P));
	    continue;
	}

	case deallocate:		/* deallocate the call record */
	    MARK(deallocate);
	    CP = E->CP;			/* unstack data on the environment */
	    CSB = E->CSB;
	    CHMAX = E->HMAX;
	    E = E->CE;
	    debug(0, numLocals = 0);
	    debug(3, printf("deallocate --- E now at 0x%05lx\n", normal(E)));
	    continue;

	case gc_allocate:		/* SPACE,REGS g/c + allocate */
	{ register cellpo ptr;
	    MARK(gc_allocate);
	    CHMAX = HMAX;
	    HMAX -= *P++;
	    if (H >= HMAX)		/* enough heap space ? */
		if (collect_garbage(*P+1) < *(P-1)) longjmp(icp_interrupt, 505);
	    P++;

	    CE = E;			/* 'save' the old environment */

	    debug(3, printf("gc_allocate --- E=0x%0l5x, B=0x%05lx, ", normal(E), normal(B)));
	    ptr = (choicepo)CE > B ? (cellpo)(CE+1)+envsize(CP) : (cellpo)(B+1);
	    if (ptr > TR - FUZZ)
		longjmp(icp_interrupt, 504);

	    ((envpo)ptr)->CP = CP;	/* store next address */
	    ((envpo)ptr)->CSB = CSB;	/* store continuation cut */
	    ((envpo)ptr)->CE = CE;	/* store old continuation environment */
	    E = (envpo)ptr;
	    ((envpo)ptr)->HMAX = CHMAX;	/* pre-allocated heap space */
	    debug(3, printf("E now at 0x%0l5x\n", normal(E)));
	    debug(0, guessLocals(P));
	    continue;
	}

	case tryme:			/* ARITY,LBL try this clause */
	    MARK(tryme);
	{
	    register cellpo ptr, Areg;
	    register int temp, argcnt;

	    Areg = A;
	    argcnt = *P++ + 1;

	    /* use the more recent of B or E */
	    ptr = (choicepo)E > B ? (cellpo)(E+1)+envsize(CP) : (cellpo)(B+1);

	    if (ptr > TR - FUZZ)
		longjmp(icp_interrupt, 504);
	    debug(3, printf("stacking %d arg regs @ Bi=0x%05lx\n", argcnt, normal(ptr)));

	    /* stack the argument registers */
	    temp = argcnt;
	    for (; temp--;)
		*ptr++ = *Areg++;

	    debug(3, printf("Bi after stacking = 0x%05lx\n", normal(ptr)));

	    ((choicepo)ptr)->AX = argcnt;
	    ((choicepo)ptr)->P = rel(P);	/* next clause to try */
	    P++;
	    ((choicepo)ptr)->CP = CP;
	    ((choicepo)ptr)->H = H;
	    ((choicepo)ptr)->B = B;
	    ((choicepo)ptr)->CSB = CSB;
	    ((choicepo)ptr)->E = E;
	    ((choicepo)ptr)->TR = TR;
	    ((choicepo)ptr)->HMAX = HMAX;

	    HB = H;
	    B = (choicepo)ptr;
	    debug2(1, printf("\ntryme --- B now at 0x%05lx\n", normal(B)));
	    debug(0, numRegs = argcnt);
	    continue;
	}


	case retryme:			/* LBL subsequent clause follows */
	    MARK(retryme);
	    B->P = rel(P);		/* next clause to try */
	    P++;
	    debug2(1, printf("\nretryme --- \n");
		      dmpA(A, TH));
	    continue;

	case trustme:			/* this is last clause to try */
	    MARK(trustme);
	    HB = B->H;			/* unstack the choice point */
	    B = B->B;
	    GC_B = min(GC_B, B);
	    debug2(1, printf("\ntrustme --- \n");
		      dmpA(A, TH));
	    continue;

	case try:			/* ARITY,LBL try a clause */
	    MARK(try);
	{
	    register cellpo ptr, Areg;
	    register int temp, argcnt;

	    argcnt = *P++ + 1;
	    Areg = A;

	    /* use the more recent of B or E */
	    ptr = (choicepo)E > B ? (cellpo)(E+1)+envsize(CP) : (cellpo)(B+1);

	    if (ptr > TR - FUZZ)
		longjmp(icp_interrupt, 504);
	    debug(3, printf("stacking %d arg regs @ Bi=0x%05lx\n", argcnt, normal(ptr)));

	    /* stack the argument registers */
	    temp = argcnt;
	    for (; temp--;)
		*ptr++ = *Areg++;

	    debug(3, printf("Bi after stacking = 0x%05lx\n", normal(ptr)));

	    ((choicepo)ptr)->AX = argcnt;	/* stack the main registers */
	    ((choicepo)ptr)->P = P + 1;		/* next clause to try */
	    ((choicepo)ptr)->CP = CP;
	    ((choicepo)ptr)->H = H;
	    ((choicepo)ptr)->B = B;
	    ((choicepo)ptr)->CSB = CSB;
	    ((choicepo)ptr)->E = E;
	    ((choicepo)ptr)->TR = TR;
	    ((choicepo)ptr)->HMAX = HMAX;

	    HB = H;
	    B = (choicepo)ptr;
	    P += *P;		/* pick up the clause to try */
	    debug2(1, printf("\ntry --- B now at 0x%05lx, P=%ld\n", normal(B), PC(P)));
	    debug(0, numRegs = argcnt);
	    continue;
	}

	case retry:			/* LBL try a subsequent clause */
	    MARK(retry);
	    B->P = P + 1;
	    P += *P;
	    debug2(1, printf("\nretry --- P=%ld\n", PC(P));
		      dmpA(A, TH));
	    continue;

	case trust:			/* LBL last clause to try */
	    MARK(trust);
	    HB = B->H;			/* unstack the choice point */
	    B = B->B;
	    GC_B = min(GC_B, B);
	    P += *P;
	    debug2(1, printf("\ntrust --- P=%ld\n", PC(P));
		      dmpA(A, TH));
	    continue;
	    
	case fdynamic:			/* s:ARITY,l:LABEL,l:BIRTH,l:DEATH */
	    MARK(fdynamic);
	{
	    register codepo	P1, P2;
	
	    P1 = P - 1;			/* point to opcode (fdynamic) */
	    findFirst(P1, C);		/* find first live clause for C (if there is one) */

	    if (P1) {
		P2 = P1;
		findNext(P2, C);	/* find next live clause for C (if there is one) */
		if (P2) {
		    register cellpo ptr, Areg;
		    register int temp, argcnt;
		    argcnt = *(P1+1) + 1;	/* number of args + one for A0 */
		    Areg = A;
		    /* use the more recent of B or E */
		    ptr = (choicepo)E > B ? (cellpo)(E+1)+envsize(CP):(cellpo)(B+1);
		    if (ptr > TR - FUZZ)
			longjmp(icp_interrupt, 504);
		    debug(3, printf("stacking %d arg regs @ Bi=0x%05lx\n", argcnt, normal(ptr)));

		    /* stack the argument registers */
		    temp = argcnt;
		    for (; temp--;)
			*ptr++ = *Areg++;

		    mkint(ptr, C);	/* stack the call clock */
		    ptr++;

		    debug(3, printf("Bi after stacking = 0x%05lx\n", normal(ptr)));

		    ((choicepo)ptr)->AX = argcnt + 1; 	/* one more for C */
		    ((choicepo)ptr)->P = P2;		/* next clause to try */
		    ((choicepo)ptr)->CP = CP;
		    ((choicepo)ptr)->H = H;
		    ((choicepo)ptr)->B = B;
		    ((choicepo)ptr)->CSB = CSB;
		    ((choicepo)ptr)->E = E;
		    ((choicepo)ptr)->TR = TR;
		    ((choicepo)ptr)->HMAX = HMAX;

		    HB = H;
		    B = (choicepo)ptr;

		    debug2(1, printf("\fdynamic --- B now at 0x%05lx\n", normal(B)));
		    debug(0, numRegs = argcnt);
		}
		P = P1 + 8;
	    }
	    else
		goto fail_label;
	    continue;
	}

	case dynamic:			/* s:ARITY,l:LABEL,l:BIRTH,l:DEATH */
	    MARK(dynamic);
	{
	    register codepo 	P1 = P - 1;	/* point to opcode (dynamic) */

	    CC = (ufourBytes)(vl(A + *P + 1));	/* restore CC */
	    findNext(P1, CC);
	    if (P1) {			/* There is a clause to be tried next */
		B->P = P1;		/* P1 points to the ARITY field of the clause */
		debug2(1, printf("\ndynamic (retry) --- \n"); dmpA(A, TH));
	    } else { 			/* No more live clauses */
		HB = B->H;		/* unstack the choice point */
		B = B->B;
		GC_B = min(GC_B, B);
		debug2(1, printf("\ndynamic (trust) --- \n"); dmpA(A, TH));
	    }
	    P += 7; /* continue at instruction following the dynamic one */
	    continue;
	}

fail_label:
	case fail:			/* fail execution */
	    MARK(fail);	
	{
	    register cellpo BAi, ptr;
	    register int temp = B->AX;
	    BAi = (cellpo)B - temp;
	    ptr = A;
	    debug(1, printf("\nfailing ...\n"));
	    debug(3, printf("unstacking %d registers from 0x%05lx\n", temp, normal(BAi)));

	    for (; temp--;)		/* restore the argument registers */
		*ptr++ = *BAi++;

	    while (TR < B->TR) {	/* reset the trailed variables */
		debug(3, printf("resetting variable @ 0x%05lx\n", normal(vl(TR))));
		mkunb((cellpo) *TR);
		TR++;
	    }
	    P   = B->P;			/* restore the stack registers */
	    CP  = B->CP;
	    H   = B->H;
	    SB  = B->B;
	    CSB = B->CSB;
	    E   = B->E;
	    HMAX = B->HMAX;
	    debug(0, numRegs = B->AX; numLocals = 0);
	    continue;
	}

	case arg_switch:		/* Ai,Lv,Li,Lf,Lc,Ln,Ll,Lt switch on argument register tag */
	    MARK(arg_switch);
	{
	    register cellpo ptr;

#ifdef INDEX_FIRST_ARG
	    ptr = A+1;
	    delnk(ptr);
	    P += (tgVal(ptr) + 1);
#else
	    ptr = &A[*P++];
	    delnk(ptr);
	    P += tgVal(ptr);
#endif

/*	interface to old tag ordering
	    switch (tgVal(ptr)) {
		case var_ref:	P += 2; break;
		case int_ref:	break;
		case float_ref: P += 2; break;
		case symb_ref:	P += 3; break;
		case nil_ref:	P += 1; break;
		case list_ref:	P += 4; break;
		case tpl_ref:	P += 5;	break;
	    }
*/
	    P += *P;
	    debug2(1, printf("\narg_switch --- tag=%d, P=%ld\n", tgVal(ptr), PC(P)));
	    continue;
	}

	case indexc:			/* Ai index constant symbol access */
	    MARK(indexc);
	{
	    register codepo probe;
	    long hsh;
	    cellpo ptr = &A[*P++];
	    int tabsize = indexlink(P);
	    delnk(ptr);
	    if ((hsh = symbhashval(symbvl(ptr)) % tabsize) < 0)
		hsh += tabsize;
	    probe = (codepo)((indexpo)P + hsh + 1);

	    for (;;) {
		register short gap;
		if (gap=indexname(probe)) {
		    if (samesymb(symbvl(ptr), (symbpo)(probe + 1 + gap)))
			break;
		    else if (gap=indexlink(probe)) {
			probe += 3 + gap;
			continue;
		    }
		}
		probe = P;
		break;
	    }
	    P = probe + 2 + indexoffset(probe);
	    continue;
	}

	case indexi:			/* Ai index integer access */
	    MARK(indexi);
	{
	    register codepo probe;
	    register long hsh;
	    register cellpo ptr = &A[*P++];
	    int tabsize = indexlink(P);
	    short gap;
	    delnk(ptr);
	    if ((hsh = intvl(ptr) % tabsize) < 0)
		hsh += tabsize;
	    probe = (codepo)((indexpo)P + hsh + 1);

	    for (;;) {
		if (intvl(ptr) == indexkey(probe))
		    break;
		else if (!indexlink(probe)) {
		    probe = P;
		    break;
		}
		else probe += indexlink(probe) + 3;
	    }

	    if (gap=indexoffset(probe))
		P = probe + 2 + gap;
	    else P += (indexoffset(P) + 2);	/* when case 0 is undefined */
	    continue;
	}

	case indext:			/* Ai,LBL,LBL index on first argument of tuple */
	    MARK(indext);
	{
	    register codepo probe;
	    register cellpo ptr = &A[*P++];
	    long hsh;
	    delnk(ptr);

	    switch (tg(functor(ptr))) {
	    case var_ref:
		P += *P;
		break;
	    case symb_ref: {
		int tabsize;
		short Ar = arity(ptr) - 1;
		P += 2;
		tabsize = indexlink(P);
		if ((hsh = symbhashval(symbvl(functor(ptr))) * Ar % tabsize) < 0)
		    hsh += tabsize;
		probe = (codepo)((indexpo)P + hsh + 1);

		for (;;) {
		    register short gap;
		    if (gap=indexname(probe)) {
			if (Ar == indexarity(probe) &&
			    samesymb(symbvl(functor(ptr)),
				     (symbpo)(probe + 1 + gap)))
			    break;
			else if (gap=indexlink(probe)) {
			    probe += 3 + gap;
			    continue;
			}
		    }
		    probe = P;
		    break;
		}
		P = probe + 2 + indexoffset(probe);
		break;
	    }
	    default:
		P++;
		P += *P;
		break;
	    }

	    continue;
	}

/* put instructions */

	case clr_y_var:         	/* Yi,Aj initialises a global variable in local */
	    MARK(clr_y_var); 
	{
	    register cellpo ptr = &(E->Y[*P++]);
	    mkunb(H);
	    *ptr = *H;
	    A[*P++] = *H++;
	    continue;
	}

	case put_x_var:			/* Xi,Aj put a new var in arg reg and temporary */
	    MARK(put_x_var);
	    mkunb(H);
	    A[*P++] = *H;
	    A[*P++] = *H++;
	    continue;

	case put_y_var:			/* Yi,Aj put a new var in arg reg and local */
	    MARK(put_y_var);
	{
	    register cellpo ptr = &(E->Y[*P++]);
	    mkunb(ptr);
	    A[*P++] = *ptr;
	    continue;
	}

	case put_x_val:         	/* Xi,Aj copy arg reg Ai to arg reg Aj */
	    MARK(put_x_val);
	{
	    register int temp = *P++;
	    A[*P++] = A[temp];
	    continue;
	}

	case put_y_val:         	/* Yi,Aj copy contents of local into arg reg */
	    MARK(put_y_val);
	{
	    register int temp = *P++;
	    A[*P++] = E->Y[temp];
	    continue;
	}

	case put_unsafe_y:      	/* Yi,Aj last occ. of an unsafe var in arg reg. */
	    MARK(put_unsafe_y);
	{
	    register cellpo ptr = &(E->Y[*P++]);
	    delnk(ptr);
	    if (IsVar(ptr) && ptr > (cellpo)E) {	/* unsafe variable */
		mkunb(H);
		mkreset(ptr);
		*ptr = *H++;
	    }
	    A[*P++] = *ptr;
	    continue;
	}

	case put_void:          	/* Ai put a void var into arg reg */
	    MARK(put_void);
	    mkunb(H);
	    A[*P++] = *H++;
	    continue;

	case put_const:         	/* C,Ai put a constant into arg reg */
	    MARK(put_const);
	{
	    register cellpo ptr;
	    register fourBytes temp = (fourBytes)rel(P);
	    P++;
	    ptr = &(A[*P++]);
	    mksymb(ptr, temp);
	    continue;
	}

	case put_int:			/* N,Ai put an integer into arg reg */
	    MARK(put_int);
	{
	    register cellpo ptr;
	    register fourBytes temp = *(fourBytes *)P;
	    P += 2;
	    ptr = &(A[*P++]);
	    mkint1(ptr, temp);
	    continue;
	}

	case put_float:			/* F,Ai put a float into arg reg */
	    MARK(put_float);
	{
	    register cellpo ptr;
	    register codepo temp = P;
	    P += 4;
	    ptr = &(A[*P++]);
	    mkcodeflt(ptr, temp);
	    continue;
	}

	case put_nil:			/* Ai put nil into arg reg */
	    MARK(put_nil);
	    mknil(&A[*P++]);
	    continue;

	case put_list:			/* Ai put list pair into arg reg */
	    MARK(put_list);
	{
	    register cellpo ptr = &(A[*P++]);
	    alloc_list(ptr, S);
	    continue;
	}

	case put_tpl:			/* ARITY,Ai put n-tuple into Ai */
	    MARK(put_tpl);
	{
	    register int temp = *P++;
	    register cellpo ptr = &(A[*P++]);
	    alloc_tpl(ptr, S, temp);
	    continue;
	}

	case put_slash:			/* Ai store slashback */
	    MARK(put_slash);
	{
	    register cellpo ptr = &(A[*P++]);
	    mkint(ptr, SB);
	    continue;
	}

/* get instructions */

	/**** NOTE: this is the same as put_x_val ********/
	case get_x_var:	        	/* Ai,Xi move arg reg */
	    MARK(get_x_var);
	{
	    register int temp = *P++;
	    A[*P++] = A[temp];
	    continue;
	}

	case get_y_var:	        	/* Ai,Yj move arg to local */
	    MARK(get_y_var);
	{
	    register int temp = *P++;
	    E->Y[*P++] = A[temp];
	    continue;
	}

	case get_x_val:	        	/* Ai,Xj unify subsequent occ. of var */
	    MARK(get_x_val);
	{
	    register cellpo ptr;
	    T1 = &A[*P++];
	    T2 = ptr = &A[*P++];
	    delnk(ptr);
	    *T2 = *ptr;			/* dereferenced result left in reg */
	    goto unify_loop;
	}

	case get_y_val:	        	/* Ai,Yj subsequent occ. of perm var */
	    MARK(get_y_val);
	    T1 = &A[*P++];
	    T2 = &(E->Y[*P++]);
	    goto unify_loop;

	case get_const:	        	/* Ai,C unify arg with a constant */
	    MARK(get_const);
	{
	    register cellpo ptr = &A[*P++];
	    register fourBytes temp = (fourBytes)rel(P);
	    delnk(ptr);
	    if (IsVar(ptr)) {
		mkreset(ptr);
		mksymb(ptr, temp);
	    } else if (NotSymb(ptr) || !samesymb(symbvl(ptr), (symbpo)temp))
		goto fail_label;
	    P++;
	    continue;
	}

	case get_int:			/* Ai,N unify arg with an integer */
	    MARK(get_int);
	{
	    register cellpo ptr = &A[*P++];
	    delnk(ptr);
	    if (IsVar(ptr)) {
		mkreset(ptr);
		mkint1(ptr, *(fourBytes *)P);
	    } else if (NotInt(ptr) || intvl(ptr) != *(fourBytes *)P)
		goto fail_label;
	    P += 2;
	    continue;
	}

	case get_float:			/* Ai,F unify arg with a float */
	    MARK(get_float);
	{
	    register cellpo ptr = &A[*P++];
	    delnk(ptr);
	    if (IsVar(ptr)) {
		mkreset(ptr);
		mkcodeflt(ptr, P);
	    } else if (NotFloat(ptr) || !numequal(floatvl(ptr), *(FLOAT *)P))
		goto fail_label;
	    P += 4;
	    continue;
	}

	case get_nil:			/* Ai unify arg with nil */
	    MARK(get_nil);
	{
	    register cellpo ptr = &(A[*P++]);
	    delnk(ptr);
	    if (IsVar(ptr)) {
		mkreset(ptr);
		mknil(ptr);
	    } else if (NotNil(ptr))
		goto fail_label;
	    continue;
	}

	case get_list:			/* Ai unify arg with a list pair */
	    MARK(get_list);
	{
	    register cellpo ptr = &(A[*P++]);
	    delnk(ptr);
	    if (IsVar(ptr)) {
		mkreset(ptr);
		alloc_list(ptr, S);
		mode = write_mode;
	    } else if (IsList(ptr)) {
		S = (cellpo) vl(ptr);	/* go into the list pair */
		mode = read_mode;
	    } else
		goto fail_label;
	    continue;
	}

	case get_tpl:			/* Ai,ARITY unify arg with a tuple */
	    MARK(get_tpl);
	{
	    register cellpo ptr = &A[*P++];
	    register int temp = *P++;
	    delnk(ptr);
	    if (IsVar(ptr)) {
		mkreset(ptr);
		alloc_tpl(ptr, S, temp);
		mode = write_mode;
	    } else if (IsTpl(ptr)) {
		S = (cellpo) vl(ptr);	/* go into the tuple */
		if (intvl(S) != temp)
		    goto fail_label;
		S++;		/* skip over the size cell of the tuple */
		mode = read_mode;
	    } else
		goto fail_label;
	    continue;
	}

/* unify instructions */

	case unify_void:		/* N skip N terms */
	    MARK(unify_void);
	    if (mode == read_mode)
		S += *P++;
	    else {
		register int temp = *P++;
		for (; temp--; S++)
		    mkunb(S);
	    }
	    continue;

	case unify_x_var:		/* Xi unify new var in structure with reg */
	    MARK(unify_x_var);
	    if (mode == write_mode)
		mkunb(S);
	    A[*P++] = *S++;
	    continue;

	case unify_y_var:		/* Yi unify new var in structure with local */
	    MARK(unify_y_var);
	{
	    register cellpo ptr = &E->Y[*P++];
	    if (mode == write_mode)
		mkunb(S);
	    *ptr = *S++;
	    continue;
	}

	case unify_x_val:		/* Xi unify var in structure with reg */
	    MARK(unify_x_val);
	    if (mode == read_mode) {
		T1 = T2 = &A[*P++];
		delnk(T1);
		*T2 = *T1;      	/* dereferenced result left in reg */
		T1 = S++;
		goto unify_loop;
	    } else
		*S++ = A[*P++];
	    continue;

	case unify_y_val:		/* Yi unify var in structure with local */
	    MARK(unify_y_val);
	    if (mode == read_mode) {
		T1 = S++;
		T2 = &E->Y[*P++];
		goto unify_loop;
	    } else
		*S++ = E->Y[*P++];
	    continue;

	case unify_loc_x_val:		/* Xi globalised unify var in structure with reg */
	    MARK(unify_loc_x_val);
	    if (mode == read_mode) {
		T1 = T2 = &A[*P++];
		delnk(T1);
		*T2 = *T1;      	/* dereferenced result left in reg */
		T1 = S++;
		goto unify_loop;
	    } else {
		register cellpo ptr = &A[*P++];
		delnk(ptr);
		if (IsVar(ptr) && ptr > BLS) {
		    mkunb(S);
		    mkreset(ptr);
		    *ptr = *S++;
		} else
			*S++ = *ptr;
	    }
	    continue;

	case unify_loc_y_val:		/* Yi globalised unify var in structure with local */
	    MARK(unify_loc_y_val);
	    if (mode == read_mode) {
		T1 = S++;
		T2 = &E->Y[*P++];
		goto unify_loop;
	    } else {
		register cellpo ptr = &E->Y[*P++];
		delnk(ptr);
		if (IsVar(ptr) && ptr > BLS) {
		    mkunb(S);
		    mkreset(ptr);
		    *ptr = *S++;
		} else
			*S++ = *ptr;
	    }
	    continue;

	case unify_const:		/* C unify against constant symbol */
	    MARK(unify_const);
	{
	    register fourBytes temp = (fourBytes)rel(P);
	    if (mode == read_mode) {
		register cellpo ptr = S++;
		delnk(ptr);
		if (IsVar(ptr)) {
		    mkreset(ptr);
		    mksymb(ptr, temp);
		}
		else if (NotSymb(ptr) || !samesymb(symbvl(ptr), (symbpo)temp))
		    goto fail_label;
	    }
	    else {
		mksymb(S, temp);
		S++;
	    }
	    P++;
	    continue;
	}

	case unify_int:	        	/* N unify against integer */
	    MARK(unify_int);
	    if (mode == read_mode) {
		register cellpo ptr = S++;
		delnk(ptr);
		if (IsVar(ptr)) {
		    mkreset(ptr);
		    mkint1(ptr, *(fourBytes *)P);
		}
		else if (NotInt(ptr) || intvl(ptr) != *(fourBytes *)P)
		    goto fail_label;
	    }
	    else {
		mkint1(S, *(fourBytes *)P);
		S++;
	    }
	    P += 2;
	    continue;

	case unify_float:	      	/* F unify against float */
	    MARK(unify_float);
	    if (mode == read_mode) {
		register cellpo ptr = S++;
		delnk(ptr);
		if (IsVar(ptr)) {
		    mkreset(ptr);
		    mkcodeflt(ptr, P);
		}
		else if (NotFloat(ptr) ||
			 !numequal(floatvl(ptr), *(FLOAT *)P))
		    goto fail_label;
	    }
	    else {
		mkcodeflt(S, P);
		S++;
	    }
	    P += 4;
	    continue;

	case unify_nil:	        	/* unify against nil */
	    MARK(unify_nil);
	    if (mode == read_mode) {
		register cellpo ptr = S++;
		delnk(ptr);
		if (IsVar(ptr)) {
		    mkreset(ptr);
		    mknil(ptr);
		}
		else if (NotNil(ptr))
		    goto fail_label;
	    }
	    else {
		mknil(S);
		S++;
	    }
	    continue;

	case unify_list:		/* unify against list */
	    MARK(unify_list);
	    if (mode == read_mode) {
		register cellpo ptr = S++;
		delnk(ptr);
		if (IsVar(ptr)) {
		    mkreset(ptr);
		    alloc_list(ptr, S);
		    mode = write_mode;
		}
		else if (IsList(ptr))
		    S = (cellpo) vl(ptr);	/* go into the list pair */
		else goto fail_label;
	    }
	    else alloc_list(S, S);
	    continue;

	case unify_tpl:	        	/* ARITY unify against tuple */
	    MARK(unify_tpl);
	{
	    register int temp;
	    if (mode == read_mode) {
		register cellpo ptr = S++;
		temp = *P++;
		delnk(ptr);
		if (IsVar(ptr)) {
		    mkreset(ptr);
		    alloc_tpl(ptr, S, temp);
		    mode = write_mode;
		}
		else if (IsTpl(ptr)) {
		    S = (cellpo) vl(ptr);	/* go into the tuple */
		    if (intvl(S) != temp)
			goto fail_label;
		    S++;		/* skip over the size cell of the tuple */
		}
		else goto fail_label;
	    }
	    else {
		temp = *P++;
		alloc_tpl(S, S, temp);
	    }
	    continue;
	}

/* set instructions */

	case set_void:			/* N skip N terms */
	    MARK(set_void);
	{
	    register int temp = *P++;
	    for (; temp--; S++)
		mkunb(S);
	    continue;
	}

	case set_x_var:			/* Xi construct new reg var */
	    MARK(set_x_var);
	    mkunb(S);
	    A[*P++] = *S++;
	    continue;

	case set_y_var:			/* Yi construct new local var */
	    MARK(set_y_var);
	{
	    register cellpo ptr = &E->Y[*P++];
	    mkunb(S);
	    *ptr = *S++;
	    continue;
	}

	case set_x_val:			/* Xi construct reg var */
	    MARK(set_x_val);
	    *S++ = A[*P++];
	    continue;

	case set_y_val:			/* Yi construct local var */
	    MARK(set_y_val);
	    *S++ = E->Y[*P++];
	    continue;

	case set_loc_x_val:		/* Xi globalised construct reg var */
	    MARK(set_loc_x_val);
	{
	    register cellpo ptr = &A[*P++];
	    delnk(ptr);
	    if (IsVar(ptr) && ptr > BLS) {
		mkunb(S);
		mkreset(ptr);
		*ptr = *S++;
	    } else
		*S++ = *ptr;
	    continue;
	}

	case set_loc_y_val:		/* Yi globalised construct local var */
	    MARK(set_loc_y_val);
	{
	    register cellpo ptr = &E->Y[*P++];
	    delnk(ptr);
	    if (IsVar(ptr) && ptr > BLS) {
		mkunb(S);
		mkreset(ptr);
		*ptr = *S++;
	    } else
		*S++ = *ptr;
	    continue;
	}

	case set_const:			/* C construct constant symbol */
	    MARK(set_const);
	{
	    register fourBytes temp = (fourBytes)rel(P);
	    mksymb(S, temp);
	    S++;
	    P++;
	    continue;
	}

	case set_int:	        	/* N construct integer */
	    MARK(set_int);
	    mkint1(S, *(fourBytes *)P);
	    S++;
	    P += 2;
	    continue;

	case set_float:	        	/* F construct float */
	    MARK(set_float);
	    mkcodeflt(S, P);
	    S++;
	    P += 4;
	    continue;

	case set_nil:	        	/* construct nil */
	    MARK(set_nil);
	    mknil(S);
	    S++;
	    continue;

	case set_list:			/* construct list */
	    MARK(set_list);
	    alloc_list(S, S);
	    continue;

	case set_tpl:	        	/* ARITY construct tuple */
	    MARK(set_tpl);
	{
	    register int temp = *P++;
	    alloc_tpl(S, S, temp);
	    continue;
	}

	case push_t:			/* push term pointer */
	    MARK(push_t);
	    pdltop->S = S + 1;
	    pdltop->mode = mode;
	    if (++pdltop > pdlmax)
		longjmp(icp_interrupt, 503);
	    continue;

	case pop_t:			/* pop term pointer */
	    MARK(pop_t);
	    pdltop--;
	    S = pdltop->S;
	    mode = pdltop->mode;
	    continue;

L_cut:
	case cut:			/* trim choice points */
	    MARK(cut);
	{
	    register cellpo tmp, ptr, dest;

	    B = SB;
	    HB = B->H;
	    GC_B = min(GC_B, B);

	    /* trim trail */
	    dest = B->TR - 1;
	    for (ptr=dest; ptr>=TR; ptr--) {
		tmp = (cellpo) vl(ptr);
		if (tmp < HB || (tmp >= BLS && tmp < (cellpo)B)) {
		    setval(dest, tmp);
		    dest--;
		}
	    }
	    TR = dest + 1;

	    debug2(1, printf("\ncut --- B now at 0x%05lx, TR=0x%05lx\n", normal(B), normal(TR)));
	    continue;
	}

	case gc:			/* SPACE,REGS garbage collection */
	    MARK(gc);
	    CHMAX = HMAX;
	    HMAX -= *P++;
	    if (H >= HMAX)		/* enough heap space ? */
		if (collect_garbage(*P+1) < *(P-1)) longjmp(icp_interrupt, 505);
	    P++;
	    continue;

	case gc0:			/* clause requires 0 heap cells */
	    MARK(gc0);
	    CHMAX = HMAX;
	    continue;

	case brkinstr:			/* breakpoint */
	    MARK(brkinstr);
#ifdef DEBUG
	    (void)printf("\n*** breakpoint ***\n\n");
	    debugLevel = oldLevel;
	    P--;
	    debugger(P);
	    Instr = get_brkcode(PC(P));
	    P++;
	    goto next_instr;
#else
	    (void)fprintf(stderr, "break entered - please recompile the emulator with DEBUG option\n");
	    longjmp(icp_interrupt, 614);
#endif

	case f_push_t:			/* first push term pointer */
	    MARK(f_push_t);
	    pdltop = pdl;
	    pdltop->S = S + 1;
	    pdltop->mode = mode;
	    pdltop++;
	    continue;

	case set_err:
		ERR_PT = rel(P);
		P++;
		continue;

	case clr_err:
		ERR_PT = (codepo) 0;
		continue;

	case go_to:
	    MARK(go_to);
	    P += *P;			/* adjust the program counter */
	    continue;
	case nop:
		MARK(nop);
		continue;
	case nop2:
		MARK(nop);
		P++;
		continue;
	case nop3:
		MARK(nop);
		P += 2;
		continue;

	case escape:			/* N escape into service function */
	    MARK(escape);
#include "escape.c"			/* ICP primitives */
	    continue;
#include "icp_low.c"

	default:			/* illegal instruction */
	    MARK(fetch_overhead);
	    (void)fprintf(stderr, "illegal instruction at address %ld\n", PC(P - 1));
	    longjmp(icp_interrupt, 614);
	}

/* this is where foreign primitives are handled */
foreign_code:
	{
	    int status = (*c_funct)();

	    switch (status) {
		case SUCCEED:	P = CP; continue;
		case FAIL:	goto fail_label;
		case REQUEUE:
		    h_deadlock = 0;
		case WAIT:
		    P = PP-1;	/* redo the foreign call when resumed */

		    /* simulate a 'resume(0)' call */
		    sP = P;		/* load shadow reg */
		    if (resume((threadpo)NULL, FALSE))
			return(-1);	/* EXIT FROM FUNCTION 'SOLVE' */
		    else if (nonempty_fdset())
			wait_for_user();
		    continue;
		    break;
		case SUSPEND:
		case SUSPEND_FOR_EVENT:
		    if (h_debug)
			(void)fprintf(stderr, "[ suspend %ld ]\n", TH);
		    P = PP-1;	/* redo the escape code when resumed */
		    if (!remove_from_runq(TH))
			continue;
		    h_deadlock = 0;
		    /* simulate a 'resume(0)' call */
		    sP = P;		/* load shadow reg */
		    if (resume((threadpo)NULL, FALSE))
			return(-1);	/* EXIT FROM FUNCTION 'SOLVE' */

		    else if (TH == prolog_th)
			longjmp(icp_interrupt, 408);

		    else {
			save_thread(TH);
			(void)fprintf(stderr, "[ forcing prolog supervisor ]\n");
			(void) resume(prolog_th, TRUE);
			return(-1);	/* EXIT FROM FUNCTION 'SOLVE' */
		    }
		    break;
		default:
		    longjmp(icp_interrupt, 409);
	    }
	}

unify_loop:
	MARK(UNIFY);
	/* out of line unification algorithm, done as a non-recursive loop */

	{
	    register
	    cellpo	Term1 = T1,
			Term2 = T2;
	    register	depth = 0;
	    cellpo	T1_stack[PDLSIZE];
	    cellpo	T2_stack[PDLSIZE];

	    while (depth >= 0) {

		if (depth > PDLSIZE)
		    longjmp(icp_interrupt, 503);

		delnk(Term1);
		delnk(Term2);

		debug(3, printf("unification - term depth is %d\n", depth));
		debug(3, printf("delinked T1= 0x%lx and T2= 0x%lx\n", normal(Term1), normal(Term2)));

		switch (tg(Term1)) {
		case var_ref:
		    MARK(UNIFY_var);
		    if (IsVar(Term2)) {	/* are we unifying two variables? */
			if (Term1 != Term2) {
			    if (Term1 < Term2) {
				debug(3, printf("binding var T2=0x%lx to T1=0x%lx\n", normal(Term2), normal(Term1)));
				mkreset(Term2);
				*Term2 = *Term1;
				break;
			    }
			    else {
				debug(3, printf("binding var T1=0x%lx to T2=0x%lx\n", normal(Term1), normal(Term2)));
				mkreset(Term1);
				*Term1 = *Term2;
				break;
			    }
			}
		    }
		    else {
			debug(3, printf("binding T1=0x%lx to T2=0x%lx\n", normal(Term1), normal(Term2)));
			mkreset(Term1);
			*Term1 = *Term2;
			break;
		    }

		case int_ref:
		    MARK(UNIFY_int);
		    if (IsVar(Term2)) {
			mkreset(Term2);
			*Term2 = *Term1;
			break;
		    } else if (NotInt(Term2) || *Term1 != *Term2)
			goto fail_label;
		    break;

		case symb_ref:
		    MARK(UNIFY_symb);
		    if (IsVar(Term2)) {
			mkreset(Term2);
			*Term2 = *Term1;
			break;
		    }
		    else if (NotSymb(Term2) || !samesymb(symbvl(Term1), symbvl(Term2)))
			goto fail_label;
		    break;

		case nil_ref:
		    MARK(UNIFY_nil);
		    if (IsVar(Term2)) {
			mkreset(Term2);
			*Term2 = *Term1;
			break;
		    }
		    else if (NotNil(Term2))
			goto fail_label;
		    break;

		case list_ref:
		    MARK(UNIFY_list);
		    if (IsVar(Term2)) {
			mkreset(Term2);
			*Term2 = *Term1;
			break;
		    }
		    else if (NotList(Term2))
			goto fail_label;
		    else {
			Term1 = (cellpo) vl(Term1);   /* stack the tails of the list */
			T1_stack[depth] = Term1 + 1;
			Term2 = (cellpo) vl(Term2);
			T2_stack[depth] = Term2 + 1;
			depth++;
			continue;      /* must'nt break 'cos Term1 and Term2 are valid */
		    }

		case tpl_ref:
		    MARK(UNIFY_tpl);
		    if (IsVar(Term2)) {
			mkreset(Term2);
			*Term2 = *Term1;
			break;
		    }
		    else if (NotTpl(Term2))
			goto fail_label;
		    else {
			register fourBytes tuple_arity;
			Term1 = (cellpo) vl(Term1);
			Term2 = (cellpo) vl(Term2);
			tuple_arity = intvl(Term1);
			if (intvl(Term2) != tuple_arity)
			    goto fail_label;
			Term1++;
			Term2++;

			for (; tuple_arity--;) {
			    T1_stack[depth] = Term1++;
			    T2_stack[depth] = Term2++;
			    depth++;
			}
			break;
		    }

		case float_ref:
		    MARK(UNIFY_float);
		    if (IsVar(Term2)) {
			mkreset(Term2);
			*Term2 = *Term1;
			break;
		    }
		    else if (NotFloat(Term2) ||
			     !numequal(floatvl(Term1), floatvl(Term2)))
			goto fail_label;
		    break;

		default:
		    MARK(UNIFY_overhead);
		    longjmp(icp_interrupt, 507);
		}

		/* come here and see what is next to do */
		if (--depth >= 0) {  /* destack Term1 and Term2 */
		    Term1 = T1_stack[depth];
		    Term2 = T2_stack[depth];
		    debug(3, printf("destacking T1 = 0x%lx, T2 = 0x%lx\n", normal(Term1), normal(Term2)));
		}
	    }
	} /* unify loop */
    } /* for */
  } /* for */
}
