/* Copyright (C) 1992 Imperial College */
/* unification sub-routines for the emulator */

#include "primitives.h"
#include "arith.h"

extern int	debugLevel;

/* compare two constants by their print name */
bool 
samesymb(A, B)
register symbpo A, B;
{
    if (A == B)
	return(SUCCEED);

    if (symbhashval(A) != symbhashval(B))
	return(FAIL);
    else {
	if (strcmp(symbname(A), symbname(B)))
	    return(FAIL);
    }
    return (SUCCEED);
}

/* alternative version :
   This uses long word comparisons, and is better for
   long constants.  However, it is slower than the simple
   version in the average case.
bool 
samesymb(A, B)
    symbpo          A, B;
{
    if (A == B)
	return(SUCCEED);

    if (symbhashval(A) != symbhashval(B))
	return(FAIL);
    else {
	register long *CA = (long *)symbname(A);
	register long *CB = (long *)symbname(B);
	register utwoBytes len = symblngth(A);
	short remainder = len & 03;
	len >>= 2;
	while (len--)
	    if (*CA++ != *CB++)
		return(FAIL);
	switch (remainder) {
	    case 1:
		return(*(char *)CA == *(char *)CB);
	    case 2:
		return(*(twoBytes *)CA == *(twoBytes *)CB);
	    case 3: {
		long mask = 0xFFFFFF00;
		return((*CA & mask) == (*CB & mask));
	    }
	}
    }
    return (SUCCEED);
}
*/



bool icp_unify(T1, T2)
register cellpo T1, T2;
{
    register int depth;
    cellpo	T1_stack[PDLSIZE];
    cellpo	T2_stack[PDLSIZE];

    depth = 0;
    while (depth >= 0) {

	if (depth > PDLSIZE)
	    longjmp(icp_interrupt, 503);

	delnk(T1);
	delnk(T2);

	debug(3, printf("unification - term depth is %d\n", depth));
	debug(3, printf("delinked T1= 0x%lx and T2= 0x%lx\n", normal(T1), normal(T2)));

	switch (tg(T1)) {
	case var_ref:
	    if (IsVar(T2)) {	/* are we unifying two variables? */
		if (T1 != T2) {
		    if (T1 < T2) {
			debug(3, printf("binding var T2=0x%lx to T1=0x%lx\n", normal(T2), normal(T1)));
			mkreset(T2);
			*T2 = *T1;
			break;
		    }
		    else {
			debug(3, printf("binding var T1=0x%lx to T2=0x%lx\n", normal(T1), normal(T2)));
			mkreset(T1);
			*T1 = *T2;
			break;
		    }
		}
	    }
	    else {
		debug(3, printf("binding T1=0x%lx to T2=0x%lx\n", normal(T1), normal(T2)));
		mkreset(T1);
		*T1 = *T2;
		break;
	    }

	case int_ref:
	    if (IsVar(T2)) {
		mkreset(T2);
		*T2 = *T1;
		break;
	    }
	    else if (NotInt(T2) || *T1 != *T2)
		return(FAIL);
	    break;

	case symb_ref:
	    if (IsVar(T2)) {
		mkreset(T2);
		*T2 = *T1;
		break;
	    }
	    else if (NotSymb(T2) || !samesymb(symbvl(T1), symbvl(T2)))
		return(FAIL);
	    break;

	case nil_ref:
	    if (IsVar(T2)) {
		mkreset(T2);
		*T2 = *T1;
		break;
	    }
	    else if (NotNil(T2))
		return(FAIL);
	    break;

	case list_ref:
	    if (IsVar(T2)) {
		mkreset(T2);
		*T2 = *T1;
		break;
	    }
	    else if (NotList(T2))
		return(FAIL);
	    else {
		T1 = (cellpo) vl(T1);   /* stack the tails of the list */
		T1_stack[depth] = T1 + 1;
		T2 = (cellpo) vl(T2);
		T2_stack[depth] = T2 + 1;
		depth++;
		continue;      /* must'nt break 'cos T1 and T2 are valid */
	    }

	case tpl_ref:
	    if (IsVar(T2)) {
		mkreset(T2);
		*T2 = *T1;
		break;
	    }
	    else if (NotTpl(T2))
		return(FAIL);
	    else {
		register fourBytes tuple_arity;
		T1 = (cellpo) vl(T1);
		T2 = (cellpo) vl(T2);
		tuple_arity = intvl(T1);
		if (intvl(T2) != tuple_arity)
		    return(FAIL);
		T1++;
		T2++;

		for (; tuple_arity--;) {
		    T1_stack[depth] = T1++;
		    T2_stack[depth] = T2++;
		    depth++;
		}
		break;
	    }

	case float_ref:
	    if (IsVar(T2)) {
		mkreset(T2);
		*T2 = *T1;
		break;
	    }
	    else if (NotFloat(T2) ||
		     !numequal(floatvl(T1), floatvl(T2)))
		return(FAIL);
	    break;

	default:
	    longjmp(icp_interrupt, 507);
	}

	/* come here and see what is next to do */
	if (--depth >= 0) {  /* destack T1 and T2 */
	    T1 = T1_stack[depth];
	    T2 = T2_stack[depth];
	    debug(3, printf("destacking T1 = 0x%lx, T2 = 0x%lx\n", normal(T1), normal(T2)));
	}
    }
    return(SUCCEED);
}
