#include <varargs.h>
#include "primitives.h"
#include "pl-itf.h"

/*  @(#) pl-itf.c 1.5.0 (UvA SWI) Jul 30, 1990

    Copyright (c) 1990 Jan Wielemaker. All rights reserved.
    See ../LICENCE to find out about your rights.
    jan@swi.psy.uva.nl

    Purpose: foreign language interface
*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
This modules  defines  the  functions  available  to  users  of  foreign
language  code.   Most  of  this  module  just is a small function layer
around primitives, normally provided via macros.   This  module  is  not
responsible for loading foreign language code (see pl-load.c). Note that
on  systems  on which pl-load.c is not portable, one can still use these
functions, link the .o files while linking prolog and call  the  foreign
module's initialisation function from main() in pl-main.c.  PCE normally
is linked this way.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

		/********************************
		*           ANALYSIS            *
		*********************************/

int
PL_type(t)
term t;
{
	delnk(t);

	if ( IsVar(t) )		return PL_VARIABLE;
	if ( IsInt(t) )		return PL_INTEGER;
	if ( IsFloat(t) )	return PL_FLOAT;
	if ( IsSymb(t) )	return PL_ATOM;
	if ( IsList(t) ||
	     IsNil(t) )		return PL_STRING;
	if ( IsTpl(t) )		return PL_TERM;

	return(FALSE);
}

double					/* double for standard arg passing */
PL_float_value(t)
term t;
{
	delnk(t);
	return floatvl(t);
}

static char pce_strings[1024];

char *
PL_string_value(lst)
term lst;
{
	char	*ptr = pce_strings;
	cellpo	el;

	delnk(lst);
	while(IsList(lst)) {
		el = hd(lst);
		delnk(el);
		if (NotInt(el) || intvl(el) < 0 || intvl(el) > 0xFF)
			return((char *)NULL);
		*ptr++ = intvl(el);
		lst = tl(lst);
		delnk(lst);
	}

	if (NotNil(lst))
		return((char *)NULL);

	*ptr = '\0';
	return(pce_strings);
}

long
PL_integer_value(t)
atomic t;
{
	delnk(t);
	return (intvl(t));
}

char *
PL_atom_value(t)
atomic t;
{
	delnk(t);
	return symbname(symbvl(t));
}

fnctor_t
PL_functor(t)
term t;
{
	delnk(t);
	return((fnctor_t)functor(t));
}

symbpo
PL_functor_name(f)
fnctor_t f;
{
	delnk(f);
	return symbvl(f);
}

int
PL_functor_arity(f)
register fnctor_t f;
{
	delnk(f);
	return arity(f) - 1;
}

term
PL_arg(t, n)
term t;
register int n;
{
	term a;

	delnk(t);
	a = arg(t, n);
	delnk(a);

	return (term) a;
}

term
PL_strip_module(t, m)
term t;
module *m;
{
	delnk(t);
	return(t);
	/* return (term) stripModule(t, m); */
}

		/********************************
		*         CONSTRUCTION          *
		*********************************/

term
PL_new_term()
{
	term var;
	alloc_cell(var);
	mkreset(var);
	mkunb(var);
	return var;
}

atomic
PL_new_atom(s)
char *s;
{
	term	var;

	alloc_cell(var);
	mkreset(var);
	alloc_symb(var, strlen(s), s);
	return (var);
}

atomic
PL_new_integer(i)
int i;
{
	term var;
	alloc_cell(var);
	mkreset(var);
	mkint(var, i);
	return (var);
}

atomic
PL_new_float(f)
double f;
{
	term var;
	alloc_cell(var);
	mkreset(var);
	alloc_float(var, f);
	return (var);
}

fnctor_t
PL_new_functor(f, a)
register atomic f;
register int a;
{
	term var, tpl;
	alloc_cell(var);
	mkreset(var);
	alloc_tpl(var, tpl, a+1);
	*tpl++ = *f;

	while(a--) {
		mkunb(tpl);
		tpl++;
	}
	return (var);
}

bool
PL_unify(t1, t2)
term t1, t2;
{ return (bool) icp_unify(t1, t2);
}

bool
PL_unify_atomic(t, w)
term t;
term w;
{ return icp_unify(t, w);
}

bool
PL_unify_functor(t, f)
term t;
fnctor_t f;
{
	delnk(t);
	delnk(f);

	if (PL_type(t) == PL_VARIABLE) {
		mklnk(t, PL_new_functor(functor(f), arity(f)-1));
		return(TRUE);
	}

	return(samesymb(symbvl(functor(t)), symbvl(functor(f))) &&
		arity(t) == arity(f));
}

		/********************************
		*        CALLING PROLOG         *
		*********************************/

extern symbpo	nested_sym;
extern codepo	search_symbol();

void
PL_mark(buf)
register bktrk_buf *buf;
{ Mark(*buf);
}

void
PL_bktrk(buf)
register bktrk_buf *buf;
{ Undo(*buf);
}

extern void timeslice();

bool
PL_call(t, m)
term t;
module m;
{
  struct thread	th;
  threadpo	original;
  bool		rval;

  timeslice(0);

  /* set sP to value of P before calling this */
  original = TH;
  save_thread(&th);
  A[1] = *t;

  rval = solve(search_symbol(nested_sym, 1));

  load_thread(&th);
  TH = original;

  timeslice(1);

  return rval;
}


#if unix
                /********************************
                *            SIGNALS            *
                *********************************/

#define MAXSIGNAL	31

void
(*PL_signal(sig, func))()
int sig;
void (*func)();
{ void (*old)();

  if ( sig < 0 || sig >= MAXSIGNAL )
  { fprintf(stderr, "PL_signal(): illegal signal number: %d", sig);
    return NULL;
  }

/*
  if ( signalHandlers[sig].catched == FALSE )
  { old = signal(sig, func);
    signalHandlers[sig].os = func;
    
    return old;
  }

  old = signalHandlers[sig].user;
  signalHandlers[sig].user = func;

  return old;
*/

  old = signal(sig, func);
  return (old);
}
#endif


		/********************************
		*           WARNINGS            *
		*********************************/

bool
vfatalError(fm, args)
char *fm;
va_list args;
{ fprintf(stderr, "[FATAL ERROR:\n\t");
  vfprintf(stderr, fm, args);
  fprintf(stderr, "]\n");
  exit(1);
}

bool
vwarning(fm, args)
char *fm;
va_list args;
{
   fprintf(stderr, "[WARNING: ");
   vfprintf(stderr, fm, args);
   fprintf(stderr, "]\n");
   return(FAIL);
}

#if defined(ANSI) && !defined(AIX) && !defined(__GNUC__)
bool
PL_warning(char *fm, ...)
{ va_list args;

  va_start(args, fm);
  vwarning(fm, args);
  va_end(args);

  fail;
}

void
PL_fatal_error(char *fm, ...)
{ va_list args;

  va_start(args, fm);
  vfatalError(fm, args);
  va_end(args);
}

#else

bool
PL_warning(va_alist)
va_dcl
{ char *fm;
  va_list args;

  va_start(args);
  fm = va_arg(args, char *);
  vwarning(fm, args);
  va_end(args);

  fail;
}

void
PL_fatal_error(va_alist)
va_dcl
{ char *fm;
  va_list args;

  va_start(args);
  fm = va_arg(args, char *);
  vfatalError(fm, args);
  va_end(args);
}
#endif ANSI

bool vPutf();

/*VARARGS1*/
bool
#if defined(ANSI) && !defined(AIX) && !defined(__GNUC__)
Putf(char *fm, ...)
{ va_list args;

  va_start(args, fm);
  vPutf(fm, args);
  va_end(args);

  succeed;
}

#else

Putf(va_alist)
va_dcl
{ va_list args;
  char *fm;

  va_start(args);
  fm = va_arg(args, char *);
  vPutf(fm, args);
  va_end(args);

  succeed;
}
#endif

bool
vPutf(fm, args)
char *fm;
va_list args;
{ char tmp[10240];
  char *s;

  vsprintf(tmp, fm, args);

  for(s=tmp; *s; s++)
    TRY(pr_write_stream(*s) );

  succeed;
}

		/********************************
		*            ACTIONS            *
		*********************************/

bool
PL_action(action, argmt)
int action;
void * argmt;
{ switch(action)
  { case PL_ACTION_TRACE:       return (bool) pl_trace();
    case PL_ACTION_DEBUG:       return (bool) pl_debug();
    case PL_ACTION_BACKTRACE:   /* backTrace(environment_frame); succeed; */
				return (bool) pl_backtrace();
    case PL_ACTION_BREAK:       return (bool) pl_break();
    case PL_ACTION_HALT:        return (bool) pl_halt();
    case PL_ACTION_ABORT:       return (bool) pl_abort();
    case PL_ACTION_SYMBOLFILE:  /* loaderstatus.symbolfile = lookupAtom((char *) argmt);
                                succeed; */
				return (bool) pl_symbolfile();
    case PL_ACTION_WRITE:       Putf("%s", (char *)argmt);
                                succeed;
    case PL_ACTION_FLUSH:       pl_flush();
                                succeed;
    default:                    fprintf(stderr, "PL_action(): Illegal action: %d", action);
                                /*NOTREACHED*/
                                fail;
  }
}

pl_trace()
{
	fprintf(stderr, "PL_ACTION : trace not implemented\n");
	succeed;
}

pl_debug()
{
	fprintf(stderr, "PL_ACTION : debug not implemented\n");
	succeed;
}

pl_break()
{
	fprintf(stderr, "PL_ACTION : break not implemented\n");
	succeed;
}

extern threadpo prolog_th;

#include <sys/ioctl.h>
pl_halt()
{
	long block = 0;
	extern double usertime();

	/* reset to blocking I/O for shell */
	(void)ioctl(0, FIONBIO, &block);
	(void) fprintf(stderr,
		"\n{ End of IC-Prolog execution, user time %.3f }\n",
		usertime() - prolog_th->stats.starttime);
	exit(0);
}

pl_abort()
{
	extern bool add_to_runq();
	extern jmp_buf icp_interrupt;

        (void) add_to_runq(prolog_th, FALSE);
        (void)fprintf(stderr, "\n--- ABORTED ---\n\n");
        longjmp(icp_interrupt, 999);
}

pl_backtrace()
{
	fprintf(stderr, "PL_ACTION : backtrace not implemented\n");
	succeed;
}

pl_symbolfile()
{
	fprintf(stderr, "PL_ACTION : symbolfile not implemented\n");
	succeed;
}
