%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%                     T H E    M A I N    L O O P
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % clauses for agent definitions
:- multifile defn/3.
:- dynamic defn/3.


% The infinite execution loop.
%
csp :- mainloop.
csp :- csp.

%
% The main driver
%

mainloop :- 
        next_expression(A),  % tokenizer and parser
        process_exp(A),      % command interpreter
        !, fail.             % recover all stack space by failure


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%   T O P    L E V E L    C O M M A N D    I N T E R P R E T E R
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% A top level command interpreter. (note: it is deterministic.)
%
process_exp(end_of_file) :- !.
process_exp(error) :- !, 
	write('*** syntax error ***'), 
	nl.
  % operator declarations
process_exp(op_defn(O,T,A,P)) :- 
        add_op(O,T,A,P), !,
        put(39),  % 39 == ~'
        write(O), 
        put(39),  % 39 == ~'
        write(' is added.'), nl.
process_exp(op_defn(O,_,_,_)) :- 
        !,
        % not add_op(O,T,A,P),
	write( '*** ' ),
        put(39),  % 39 == ~'
        write(O), 
        put(39),  % 39 == ~'
        write(' is already defined.'), 
	write( ' ***' ),
	nl.
  % saving prolog state
process_exp(func(lid(save),1,[File])) :-
        !,
	file_name(File,Name),
        save(Name),
	banner.
  % saving definitions
process_exp(func(lid(save_defs),1,[File])) :-
        !,
        save_defs_file(File).
  % saving operators
process_exp(func(lid(save_ops),1,[File])) :-
        !,
        save_ops_file(File).
  % loading files
process_exp(func(lid(load),1,[File])) :-
        !,
        load_file(File).
  % stepping agents
process_exp(func(lid(step),1,[Agent])) :-
        !,
        semantic_check_agent(Agent,Agent1,[]),
        step_agent(Agent1).
  % tracing agents
process_exp(func(lid(traces),2,[Agent,num(N)])) :-
        !,
        semantic_check_agent(Agent,Agent1,[]),
        traces(Agent1,N,L),
	print_traces( L ).
  % running agents
process_exp(func(lid(runs),2,[Agent,num(N)])) :-
        !,
        semantic_check_agent(Agent,Agent1,[]),
        runs(Agent1,N,L),
	print_traces( L ).
  % stepping agents
process_exp(func(lid(step),2,[Agent,num(N)])) :-
        !,
        semantic_check_agent(Agent,Agent1,[]),
        run(Agent1,1,N).
  % model checking 
process_exp(op('|=',Agent,Property)) :-
        semantic_check_agent(Agent,Agent1,[]),
        semantic_check_formula(Property,Property1,[]),
	(satisfy( Property1, Agent1, [], [], 0, _ )->
	    (write( 'true' ), nl)
	  ; (write( 'false' ), nl) ).
  % deleting
process_exp(func(lid(delete),1,[Agent])) :-
        !,
        delete_agent(Agent).
  % listing
process_exp(func(lid(list),1,[lid(all)])) :-
        !,
        list_defns.
process_exp(func(lid(list),1,[Agent])) :-
	defn(Agent,Exp,_),
	print_expression(op('::=',Agent,Exp)), nl, 
	fail.     % next one
process_exp(func(lid(list),1,[_])) :- !.
  % stopping
process_exp(lid(quit)) :- !, halt.
process_exp(lid(halt)) :- !, halt.
process_exp(lid(abort)) :- !, abort.
  % helping
process_exp(lid(help)) :- !,
%	write( '(infix | prefix | postfix)' ),
%       write( ' (non | right | left) (0..1200) <Op>.' ), nl,
        write( '<Const> ::= ( <Agent> | <Formula> ).' ), nl,
	write( 'delete( <Agent> ) | delete(all).' ), nl,
	write( 'list( <Agent> ) | list(all).' ), nl,
	write( 'load( <File> ).' ), nl,
	write( 'lts( <Agent> ).' ), nl,
	write( 'min( <Agent> ).' ), nl,
	write( 'minlts( <Agent> ).' ), nl,
	write( 'quit | halt.' ), nl,
	write( 'runs( <Agent>, <Num> ).' ), nl,
	write( 'save_ops( <File> ).' ), nl,
	write( 'save_defs( <File> ).' ), nl,
	write( 'step( <Agent> ) | step( <Agent>, <Num> ).' ), nl,
	write( 'traces( <Agent>, <Num> ).' ), nl,
	write( '<Agent> =a= <Agent>.' ), nl,
	write( '<Agent> =l= <Agent>.' ), nl,
	write( '<Agent> |= <Formula>.' ), nl.
  % constructing the labelled transition system
process_exp(func(lid(lts),1,[P1])) :-
	!,
        semantic_check_agent(P1,T1,[]),
	write( 'LTS of ' ), print_expression(T1), write( ':' ), nl,
	lts(T1,LTS1,States1),
	show_lts(LTS1,States1).
  % bisimulation checking on the explicitly constructed LTSs
process_exp(op('=l=',P1,P2)) :- 
	!,
        semantic_check_agent(P1,T1,[]),
        semantic_check_agent(P2,T2,[]),
	lts(T1,LTS1,_),
	lts(T2,LTS2,_),
	(sbisim_lts(LTS1,LTS2,S)->
	    (write( 'A strong bisimulation exists : ' ), 
	     nl, show_bisim(S), write( 'with ' ),
	     length( S, N ), write( N ), write( ' classes.' ), nl)
	  ; (write( '*** not bisimilar ***'), nl) ).
  % bisimulation checking on the agents
process_exp(op('=a=',P1,P2)) :- 
	!,
        semantic_check_agent(P1,T1,[]),
        semantic_check_agent(P2,T2,[]),
	(sbisim(T1,T2,S)->
	    (write( 'A strong bisimulation exists : ' ), 
	     nl, show_bisim(S), write( 'with ' ),
	     length( S, N ), write( N ), write( ' classes.' ), nl)
	  ; (write( '*** not bisimilar ***'), nl) ).
  % minimization of agents
process_exp(func(lid(minlts),1,[P])) :-
	!,
        semantic_check_agent(P,T,[]),
	lts(T,LTS,_),
	sbisim_lts(LTS,LTS,S),
	write( 'Its minimized LTS is : ' ), nl, show_fa(S,0). 
process_exp( op(min,P) ) :-
	!,
        semantic_check_agent(P,T,[]),
	sbisim(T,T,S),
	write( 'Its minimized agent is : ' ), nl, show_fa(S,0). 
  % introducing new definitions
process_exp(op('::=',L,R)) :- 
	!,
        semantic_check(op('::=',L,R),op('::=',L1,R1),Vars),
        print_expression(L1), 
        nl,
        save_defn(op('::=',L1,R1),Vars).
process_exp( A ) :-
        write('*** invalid command *** '),
        print_expression( A ), 
	nl.
        
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%                A U X I L I A R Y     P R E D I C A T E S
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%% delete(Const)
%	removes the definitions of "Const" permanently
%
delete_agent( lid(all) ) :-
        !,
	repeat,
	  (defn(Agent,_,_)->
	      (retractall(defn(Agent,_,_)), fail)
	    ; true).
delete_agent( Agent ) :-
        defn(Agent,_,_),
        !,
        print_expression(Agent),
        retractall(defn(Agent,_,_)),
	write( ' is deleted.' ),
        nl.
delete_agent( Agent ) :-
        print_expression(Agent),
        write(' does not exist.'), nl.

  % listing all definitions
list_defns :-
        defn(Agent,Defn,_),
        print_expression(op('::=',Agent,Defn)), nl,
        fail.
list_defns.

  % storing definitions
save_defn( op('::=',A,D), Vars) :-
        assert( defn(A,D,Vars) ).

  % saving operator declarations in a file
save_ops_file(Name) :-
        file_name(Name,File), !,
        save_ops(File).

  % saving definitions in a file
save_defs_file(Name) :-
        file_name(Name,File), !,
        save_defs(File).

file_name(lid(File),  File).
file_name(uid(File),  File).
file_name(const(File),File).

save_defs(File) :-
        telling(Cur),
        tell(File),
	saving,
        told,
	tell(Cur).

save_ops(File) :- 
        telling(Cur),
        tell(File),
        listing(operatorN),
        listing(operator3),
        listing(operator2),
        listing(operator1),
        listing(op_decl),
        told,
        tell(Cur).

saving :-
	repeat,
	  defn(A,B,_),
	  print_expression(op('::=', A, B)),
	fail.
saving.
	
  % loading a file
load_file(Name) :-
        file_name(Name,File), !,
        load_fl(File).

load_fl(File) :-
        write('Loading file: '),
        put(39),  % 39 == ~'
        write(File),
        put(39),  % 39 == ~'
        nl, nl,
        prompt_off,
        seeing(F),
        see(File),
        loading,
        seen,
        nl,
        put(39),  % 39 == ~'
        write(File),
        put(39),  % 39 == ~'
        write(' is loaded.'),
        nl,
        see(F),
        reset_prompt.

loading :-
        repeat,
          load_expression, 
        !.

%% load_expression succeeds if end of file is reached
%       otherwise it always fails
%
load_expression :-
        next_expression(A),
        A \== end_of_file,
        process_exp(A), 
        !, fail.
load_expression.        % end of file is reached

reset_prompt :-
        seeing(F),      % just in case the loading is nested
        F = user, !,    % if we are reading the terminal
        prompt_on.
reset_prompt.
