%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%                V E R I F I C A T I O N     C H E C K E R 
%
%                 Author: Mantis H.M. Cheng (Jun/13/1994)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%                    M O D A L    M U - C A L C U L U S
%
%              (Tableau Property Checker for Finite-State Systems)
%                ( with a fixed-point rule but no thinning rule )
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% satisfy( +Form, +Agent, +Def, +Sat, +Count, -Count1 )
%
%	Form = a modal (mu-calculus) formula
%	Agent = an agent to be shown to satisfy "Form"
%	Def = a list of definitions for fixed-point unfolding
%	Sat = a list of pairs (P,E) each denoting P |= E has been proven
%	Count,Count1 = counter for introducing new definitional constants
%
  %
  % P |= tt   for all P
  %
satisfy( form(tt), _, _, _, N, N ) :- !.
  %
  % P |=/= ff   for all P
  %
satisfy( form(ff), _, _, _, _, _ ) :-
	!, 
	fail.
  %
  % P |= not F  iff  P |=/= F
  %
satisfy( op('not',Prop), P, Def, Sat, N, N ) :- !,
	\+ satisfy( Prop, P, Def, Sat, N, _ ).
  %
  % P |= F1 and F2   iff  P |= F1 and P |= F2
  %
satisfy( op('and',Prop1,Prop2), P, Def, Sat, N, N2 ) :- !,
	satisfy( Prop1, P, Def, Sat, N,  N1 ),
	satisfy( Prop2, P, Def, Sat, N1, N2 ).
  %
  % P |= F1 or F2   iff  P |= F1 or P |= F2
  %
satisfy( op('or',Prop1,_), P, Def, Sat, N, N1 ) :-
	satisfy( Prop1, P, Def, Sat, N, N1 ).
satisfy( op('or',_,Prop2), P, Def, Sat, N, N1 ) :- !,
	satisfy( Prop2, P, Def, Sat, N, N1 ).
  %
  % P |= U  iff U ::= max Z.E(Z) or min Z.E(Z), U has not been satisfied
  %             before, if so, then it is true for 'max' and false for
  %             'min'; otherwise, P |= E(Z) { U/Z } and remember that
  %             P |= U.
  %
satisfy( num(C), P, Def, Sat, N, N1 ) :- !, 
	member( num(C)=op('.',op(O,Z),E), Def ),
	(member( sat(P,num(C)), Sat )->
		(O = 'max' -> true ; fail)
	; ( replace( Z, E, num(C), E1 ),
	    satisfy( E1, P, Def, [sat(P,num(C))|Sat], N, N1 ) )).
  %
  % P |= D  iff D ::= F and P |= F
  %
satisfy( uid(C), P, Def, Sat, N, N1 ) :- !,                % definition
	defn( uid(C), Prop, _ ),
	satisfy( Prop, P, Def, Sat, N, N1 ).
satisfy( func(uid(C),M,T), P, Def, Sat, N, N1 ) :- !,      % definition
	eval( func(uid(C),M,T), Form ),
	defn( Form, Prop, _ ),
	satisfy( Prop, P, Def, Sat, N, N1 ).
  %
  % P |= O Z . E(Z)  iff  P |= U where U = O Z . E(Z), and O is 'max' or 'min'
  %
satisfy( op('.',op(O,Z),E), P, Def, Sat, N, N2 ) :- 
	member( O, ['min','max'] ), !,
	N1 is N+1, 
	satisfy(num(N),P,[num(N)=op('.',op(O,Z),E)|Def], Sat, N1, N2 ).
  %
  % P |= [[ - ]].E     iff R |= E for all R in { Q : P ==a==> Q, any a }
  %
satisfy( op('.',box('-'),E), P, Def, Sat, N, N1 ) :- !,
	successors( P, Succs ),
	all_satisfy( Succs, E, Def, Sat, N, N1 ).
  %
  % P |= [[ K ]].E     iff R |= E for all R in { Q : P ==a==> Q, a in K }
  %
satisfy( op('.',box(set(K)),E), P, Def, Sat, N, N1 ) :- !,
	successors( P, Succs ),
	partition_successors( Succs, K, SuccsK, _ ),
	all_satisfy( SuccsK, E, Def, Sat, N, N1 ).
  %
  % P |= [[-K ]].E     iff R |= E for all R in { Q : P ==a==> Q, a not in K }
  %
satisfy( op('.',box(op('-',set(K))),E), P, Def, Sat, N, N1 ) :- !,
	successors( P, Succs ),
	partition_successors( Succs, K, _, SuccsNotK ),
	all_satisfy( SuccsNotK, E, Def, Sat, N, N1 ).
  %
  % P |= [[ a ]].E     iff P |= [[ {a} ]].E
  %
satisfy( op('.',box(Act),E), P, Def, Sat, N, N1 ) :-
	pure_action( Act ), !,
	findall( pr(Act,Q), trans(P,Act,Q), Succs ),
	all_satisfy( Succs, E, Def, Sat, N, N1 ).
  %
  % P |= [[ -a ]].E     iff P |= [[ -{a} ]].E
  %
satisfy( op('.',box(op('-',Act)),E), P, Def, Sat, N, N1 ) :-
	pure_action( Act ), !,
	successors( P, Succs ),
	partition_successors( Succs, [Act], _, SuccsNotAct ),
	all_satisfy( SuccsNotAct, E, Def, Sat, N, N1 ).
  %
  % P |= [[ [a,b,c] ]].E  = P |= [[a]].[[b]].[[c]].E
  %
satisfy( op('.',box([A]),E), P, Def, Sat, N, N1 ) :-
	pure_action( A ), !,
	satisfy( op('.',box(A),E), P, Def, Sat, N, N1 ).
satisfy( op('.',box([A1,A2|As]),E), P, Def, Sat, N, N1 ) :-
	pure_action( A1 ), !,
	satisfy( op('.',box(A1),op('.',box([A2|As]),E)), P, Def, Sat, N, N1 ).
  %
  % P |= << - >>.E     iff R |= E for some R in { Q : P ==a==> Q }
  %
satisfy( op('.',diamond('-'),E), P, Def, Sat, N, N1 ) :- !,
	trans( P, _, Q ),
	satisfy( E, Q, Def, Sat, N, N1 ).
  %
  % P |= << K >>.E     iff R |= E for some R in { Q : P ==a==> Q, a in K }
  %
satisfy( op('.',diamond(set(K)),E), P, Def, Sat, N, N1 ) :- !,
	member( Act, K ),
	trans( P, Act, Q ),
	satisfy( E, Q, Def, Sat, N, N1 ).
  %
  % P |= << -K >>.E     iff R |= E for some R in { Q : P ==a==> Q, a not in K }
  %
satisfy( op('.',diamond(op('-',set(K))),E), P, Def, Sat, N, N1 ) :- !,
	trans( P, Act, Q ),
	non_member( Act, K ),
	satisfy( E, Q, Def, Sat, N, N1 ).
  %
  % P |= << a >>.E   iff P |= << {a} >>.E
  %
satisfy( op('.',diamond(Act),E), P, Def, Sat, N, N1 ) :-
	pure_action( Act ), !,
	trans( P, Act, Q ),
	satisfy( E, Q, Def, Sat, N, N1 ).
  %
  % P |= << -a >>.E  iff  P  |= << -{a} >>.E
  %
satisfy( op('.',diamond(op('-',Act)),E), P, Def, Sat, N, N1 ) :-
	pure_action( Act ), !,
	trans( P, Act1, Q ),
	Act1 \== Act,
	satisfy( E, Q, Def, Sat, N, N1 ).
  %
  % P |= << [a,b,c] >>.E  = P |= <<a>>.<<b>>.<<c>>.E
  %
satisfy( op('.',diamond([A]),E), P, Def, Sat, N, N1 ) :-
	pure_action( A ), !,
	satisfy( op('.',diamond(A),E), P, Def, Sat, N, N1 ).
satisfy( op('.',diamond([A1,A2|As]),E), P, Def, Sat, N, N1 ) :-
	pure_action( A1 ), !,
	satisfy( op('.',diamond(A1),op('.',diamond([A2|As]),E)), P, 
                 Def, Sat, N, N1).


%% partition_successors( +Succs, +K, -SuccsK, -SuccsNotK )
%	holds if "SuccsK" is the set of successors pr(A,Q) in "Succs", 
%	where A is in "K", and "SuccsNotK" is the rest.
%
partition_successors( [],              _, [],     []        ) :- !.
partition_successors( [pr(A,Q)|Succs], K, SuccsK, SuccsNotK ) :- !,
	(member( A, K )->
		(SuccsK = [pr(A,Q)|SuccsK1],
		 partition_successors( Succs, K, SuccsK1, SuccsNotK ))
	; 	(SuccsNotK = [pr(A,Q)|SuccsNotK1],
		 partition_successors( Succs, K, SuccsK, SuccsNotK1 )) ).


%% all_satisfy( +Succs, +E, +Def, +Sat, +Count, -Count1 )
%	holds just in case when all successors in "Succs" satisfy "E".
%	Note: "ff" is only satisfiable by an empty set of successors.
%
all_satisfy( [],                   form(ff), _,  _  , N, N ) :- !.
all_satisfy( [pr(_,Q)],            E,       Def, Sat, N, N1 ) :- !,
	satisfy( E, Q, Def, Sat, N, N1 ).
all_satisfy( [pr(_,Q),Succ|Succs], E,       Def, Sat, N, N2 ) :-
	satisfy( E, Q, Def, Sat, N, N1 ),
	all_satisfy( [Succ|Succs], E, Def, Sat, N1, N2 ).



%% replace( +Z, +E, +R, -E1 ).
%	"E1" is the formula after replacing every free occurrance of "Z"
%	in "E" by "R", i.e., E1 = E { R/Z }.
%
replace( Z,  Z,  E,  E ) :- !.
replace( Z, op(O,L,R), E, op(O,L1,R1) ) :-
        member( O, ['and','or'] ), !,
	replace(Z,L,E,L1),
	replace(Z,R,E,R1).
replace( Z, op('.',op(C,X),R), E, op('.',op(C,X),R1) ) :-
	member( C, ['min', 'max'] ), 
	Z \== X, !,
	replace(Z,R,E,R1).
replace( Z, op('.',box(M),R), E, op('.',box(M),R1) ) :- !,
	replace(Z,R,E,R1).
replace( Z, op('.',diamond(M),R), E, op('.',diamond(M),R1) ) :- !,
	replace(Z,R,E,R1).
replace( _, E, _, E ).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%          L A B E L L E D    T R A N S I T I O N    S Y S T E M S
%
%                        (construction)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%% lts( Agent, LTS, States )
%	"LTS" is the labelled transition system associated with "Agent".
%	It is a set of transitions of the form (Si,Action,Sj) where
%	Si and Sj are state numbers. The "Agent" is always denoted by
%	the number 0. "States" is a list of pairs "pr(Agent,Num)" where
%	"Num" is the state number associated with "Agent".
%
lts( Agent, LTS1, States1 ) :-
	simplify(Agent,Av),
	findall( trans(Av,Act,Agent1),
	         trans(Av,Act,Agent1), L ),
 	mklts( L, [], LTS, [pr(Av,0)], States, 1 ),
	reverse( LTS, LTS1 ),
	reverse( States, States1 ).

mklts( [], L, L, As, As, _ ) :- !.
mklts( [trans(P,A,P1)|Ts], L, LTS, As, As2, N ) :-
	instantiate(A),
	simplify(P1,P1v),
	same_member( pr(P,PN), As ),
	new_name(P1v,P1N,As,As1,N,N1),
	(\+ member( trans(PN,A,P1N), L )->
		(findall( trans(P1v,A2,P2), trans(P1v,A2,P2), L2 ),
		 append( Ts, L2, Ts1 ),
		 mklts( Ts1, [trans(PN,A,P1N)|L], LTS, As1, As2, N1 )) ;
		mklts( Ts, L, LTS, As1, As2, N1 ) ).


new_name( P, N, L, L1, C, C1 ) :-
	(same_member( pr(P,N), L )->
		(C = C1, L = L1) ;
		(N = C, L1 = [pr(P,C)|L], C1 is C+1) ).


same_member( pr(P,N), [pr(P1,N)|_] ) :- equal( P, P1 ), !.
same_member( P,       [_|Ps]       ) :- same_member( P, Ps ).
	

%% show_lts( LTS, States )
%	display the labelled transition system in a linear order
%
show_lts( [], States ) :- !,
	write( 'where' ), nl,
	show_states( States ).
show_lts( [trans(P,A,P1)|Ts], States ) :-
	write( '  ' ),
	write(P),
	write( ' == ' ),
	print_expression(A),
	write( ' ==> ' ),
	write(P1),
	nl,
	show_lts(Ts, States).

show_states( [] ) :- !.
show_states( [pr(P,N)|Ts] ) :-
	write( '  ' ),
	write(N),
	write( ' = ' ),
	print_expression(P),
	nl,
	show_states(Ts).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%                   S T R O N G     B I S I M U L A T I O N
%
%                             (on agents)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%% sbisim( +Agent1, +Agent, -BS )
%	holds if "Agent1" and "Agent2" are strongly bisimilar with a
%	a strong bisimulation "BS"
%	(N.B. It is different from sbisim_lts/3. The calculation starts
%	      with two agents, instead of two LTSs. However, the techniques
%	      are identical.)
%
sbisim( P1, P2, S1 ) :-
	sbisim1( P1, P2, [], S ),
	partition( S, S1 ).


%% sbisim1( P1, P2, S1, S3 )
%	The agents P1 and P2 are strongly bisimilar with S1 being the 
%	initial and S3 the final bisimulation.
%
sbisim1( P1, P2, S1, S3 ) :-
	eval( P1, P1v ),
	eval( P2, P2v ),
	(member_class( P1v-P2v, S1 ) ->
	  (S1 = S3)
	; (successors( P1v, Succ1 ),
	   successors( P2v, Succ2 ),
	   add_member_class( P1v-P2v, S1, S2 ),
	   sbisim_successors( Succ1, Succ2, S2, S3 ))).

%% sbisim_successors( Succ1, Succ2, S1, S3 )
%	The successors Succ1 and Succ2 can strongly simulate each other,
%	with S1 being the initial and S3 the final bisimulation.
%
sbisim_successors( []   , []   , S , S  ) :- !.
sbisim_successors( Succ1, Succ2, S1, S3 ) :-
	sim_successors1( Succ1, Succ2, S1, S2 ),
	sim_successors2( Succ1, Succ2, S2, S3 ).


%% sbisim_successors1( Succ1, Succ2, S1, S3 )
%	The successors Succ1 can be simulated by the successors Succ2,
%	with S1 being the initial and S3 the final bisimulation.
%
sim_successors1( []    , _     , S , S  ) :- !.
sim_successors1( [pr(A,P1)|Succs1], Succs2, S1, S3 ) :-
	member( pr(A,P2), Succs2 ),
	sbisim1( P1, P2, S1, S2 ),
	sim_successors1( Succs1, Succs2, S2, S3 ).


%% sbisim_successors2( Succ1, Succ2, S1, S3 )
%	The successors Succ2 can be simulated by the successors Succ1,
%	with S1 being the initial and S3 the final bisimulation.
%
sim_successors2( _     , []    , S , S  ) :- !.
sim_successors2( Succs1, [pr(A,P2)|Succs2], S1, S3 ) :-
	member( pr(A,P1), Succs1 ),
	sbisim1( P1, P2, S1, S2 ),
	sim_successors2( Succs1, Succs2, S2, S3 ).


%% show_bisim( +BS )
%	display the bisimulation "BS"
%
show_bisim( [] ) :- !.
show_bisim( [pr(X,Y)|Cs] ) :-
	write( '  ' ),
	print_expression(set(X)), 
	write( ' = ' ),
	print_expression(set(Y)), 
	nl,
	show_bisim(Cs).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%                   S T R O N G     B I S I M U L A T I O N
%
%                       (on Labelled Transition Systems)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%% sbisim_lts( +LTS1, +LTS2, -S )
%	holds if "LTS1" and "LTS2" are strongly bisimilar with a strong
%	bisimulation "S", assuming their initial states are (0,0).
%
sbisim_lts( LTS1, LTS2, S ) :-
	sbisim_lts1( 0-0, LTS1, LTS2, [], BS1 ),
	partition( BS1, S ).


%% sbisim_lts1( P1-P2, LTS1, LTS2, S1, S3 )
%	The agents P1 and P2 are strongly bisimilar with S1 being the 
%	initial and S3 the final bisimulation.
%
sbisim_lts1( P1-P2, LTS1, LTS2, S1, S3 ) :-
	(member_class( P1-P2, S1 ) ->
	  (S1 = S3)
	; (findall( pr(A1,P11), member( trans(P1,A1,P11), LTS1 ), Succ1 ),
	   findall( pr(A2,P22), member( trans(P2,A2,P22), LTS2 ), Succ2 ),
	   add_member_class( P1-P2, S1, S2 ),
	   sbisim_lts_successors( Succ1, Succ2, LTS1, LTS2, S2, S3))).

%% sbisim_lts_successors( Succ1, Succ2, LTS1, LTS2, S1, S3 )
%	The successors Succ1 and Succ2 can strongly simulate each other,
%	with S1 being the initial and S3 the final bisimulation.
%
sbisim_lts_successors( []   , []   , _   , _   , S , S  ) :- !.
sbisim_lts_successors( Succ1, Succ2, LTS1, LTS2, S1, S3 ) :-
	sim_lts_successors1( Succ1, Succ2, LTS1, LTS2, S1, S2 ),
	sim_lts_successors2( Succ1, Succ2, LTS1, LTS2, S2, S3 ).


%% sbisim_lts_successors1( Succ1, Succ2, LTS1, LTS2, S1, S3 )
%	The successors Succ1 can be simulated by the successors Succ2,
%	with S1 being the initial and S3 the final bisimulation.
%
sim_lts_successors1( []    , _     , _   , _   , S , S  ) :- !.
sim_lts_successors1( [pr(A,P1)|Succs1], Succs2, LTS1, LTS2, S1, S3 ) :-
	member( pr(A,P2), Succs2 ),
	sbisim_lts1( P1-P2, LTS1, LTS2, S1, S2 ),
	sim_lts_successors1( Succs1, Succs2, LTS1, LTS2, S2, S3 ).


%% sbisim_lts_successors2( Succ1, Succ2, LTS1, LTS2, S1, S3 )
%	The successors Succ2 can be simulated by the successors Succ1,
%	with S1 being the initial and S3 the final bisimulation.
%
sim_lts_successors2( _     , []    , _   , _   , S , S  ) :- !.
sim_lts_successors2( Succs1, [pr(A,P2)|Succs2], LTS1, LTS2, S1, S3 ) :-
	member( pr(A,P1), Succs1 ),
	sbisim_lts1( P1-P2, LTS1, LTS2, S1, S2 ),
	sim_lts_successors2( Succs1, Succs2, LTS1, LTS2, S2, S3 ).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%              A U X I L A R Y      P R E D I C A T E S
%
%                    (for bisimulation checking)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%% show_fa( +BS, +N )
%	display the states of a strong bisimulation
%
show_fa( [], N ) :- !,
	write( 'with ' ),
        write( N ), 
	write( ' states.' ), nl.
show_fa( [pr(X,_)|Cs], N ) :-
	write( '  ' ),
	print_expression(set(X)),
	nl,
	N1 is N+1,
	show_fa(Cs,N1).


%% member_class( +P, +BS1 )
%	holds if the element "P" is an "identical" member of one of the 
%	equivalence classes of "BS1"
%	"BS1" is a list of equivalence classes, each of which is a list 
%	of elements of the form X-Y.
member_class( P, [P1|Ps] ) :-
	member( P, P1 )-> true ; member_class( P, Ps ).


%% add_member_class( +P, +BS1, -BS2 )
%	holds if the element "P" can be added to one of the equivalence
%	classes of "BS1" to produce "BS2", or be added as a new singleton
%	class. "P" is assumed to be not in "BS1".
%	"BS1" is a list of equivalenc classes, each of which is a list 
%	of elements of the form X-Y.
%
add_member_class( P, BS1, [EQ|NEQs] ) :-
	select_equal_classes( P, BS1, [P], EQ, NEQs ).


%% select_equal_classes( +P, +BS, +EQ1, -EQ2, -NEQ )
%	select from those equivalence classes of "BS" such that they are
%	equivalent to "P" given "EQ1" is the initial class that is equivalent
%	to "P"; "EQ2" is the final class equivalent to "P" and "NEQ" is the
%	remaining set of classes from "BS" that are not equivalent to "P".
%	Note: "EQ2" is a single equivalence class, while "NEQ" is a set of
%	      equivalence classes. We merge all those classes that are
%	      equivalent to "P" into one.
%
select_equal_classes( _, [],      EQ,  EQ,  [] ) :- !.
select_equal_classes( P, [P1|Ps], EQ1, EQ3, NEQs ) :-
	(same_class( P, P1 )->
	   (append( P1, EQ1, EQ2 ), 
	    select_equal_classes( P, Ps, EQ2, EQ3, NEQs ))
	 ; (NEQs = [P1|NEQs1], 
	    select_equal_classes( P, Ps, EQ1, EQ3, NEQs1 )) ).


%% same_class( +(X-Y), +C )
%	holds if the pair "X-Y" is in the class "C"
%
same_class( X-_, [X1-_|_] ) :- equal( X, X1 ), !.
same_class( _-Y, [_-Y1|_] ) :- equal( Y, Y1 ), !.
same_class( P,   [_|Ps] ) :- 
	same_class( P, Ps ).


equal( P, P ) :- !.
equal( P, Q ) :- defn( P, Q, _ ), !.
equal( P, Q ) :- defn( Q, P, _ ), !.


%% add_element( +P, +L1, -L2 )
%	the element "P" is added into the list "L1" to produce "L2";
%	no two "P"s are added twice.
%
add_element( P, []      ,  [P]      ) :- !.
add_element( P, [P1|Ps1] , [P1|Ps2]  ) :-
	(P = P1 ->
	   Ps1 = Ps2
	 ; add_element( P, Ps1, Ps2)).


%% partition( +BS1, -BS2 )
%	equivalent pairs of states in the strong bisimulation "BS1" are
%	combined into "BS2"
%
partition( [], [] ) :- !.
partition( [P|Ps], [pr(PX,PY)|Ps1] ) :-
	collectXY( P, [], PX, [], PY ),
	partition( Ps, Ps1 ).


collectXY( [],       Xs, Xs,  Ys, Ys ) :- !.
collectXY( [X-Y|Ps], Xs, Xs2, Ys, Ys2 ) :-
	add_element( X, Xs, Xs1 ),
	add_element( Y, Ys, Ys1 ),
	collectXY( Ps, Xs1, Xs2, Ys1, Ys2 ).

